// Copyright (C) 2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file concepts.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief concepts - experimental
 * 
 * based on https://www.youtube.com/watch?v=VctviQl-SR4
 * and https://github.com/slurps-mad-rips/cxx-concepts
 */

#ifndef NV_STL_TYPE_TRAITS_CONCEPTS_HH
#define NV_STL_TYPE_TRAITS_CONCEPTS_HH

#include <nv/stl/type_traits.hh>
#include <nv/stl/type_traits/experimental.hh>

namespace nv
{
	
	template < typename T, typename Void, template < typename...> class, typename...>
	struct detector : identity<T> { using value_t = false_type; };

	template < typename T, template < typename... > class U, typename... Args>
	struct detector<T, void_t< U<Args...> >, U, Args... >
		: identity< U< Args... > >
	{
		using value_t = true_type;
	};

	struct nonesuch final
	{
		nonesuch( nonesuch const& ) = delete;
		nonesuch() = delete;
		~nonesuch() = delete;
		void operator = ( nonesuch const& ) = delete;
	};

	template < typename T, template < typename... > class U, typename... Args >
	using detected_or = detector<T, void, U, Args...>;

	template < template < typename... > class T, typename... Args >
	using detected_t = typename detected_or<nonesuch, T, Args...>::type;

	template < typename T, template < typename... > class U, typename... Args>
	using detected_or_t = typename detected_or<T, U, Args...>::type;

	template < typename To, template < typename... > class T, typename... Args>
	using is_detected_convertible = is_convertible<
		detected_t<T, Args...>,
		To
	>;

	template < typename T, template < typename... > class U, typename... Args >
	using is_detected_exact = is_same<T, detected_t<U, Args...>>;

	template < template < typename... > class T, typename... Args >
	using is_detected = typename detected_or<nonesuch, T, Args...>::value_t;

	template < typename... > struct conjunction;
	template < typename... > struct disjunction;

	template < typename B > 
	using negation = bool_constant< !B::value >;

	template < typename T, typename... Ts>
	struct conjunction<T, Ts...> :
		bool_constant<T::value && conjunction<Ts...>::value>
	{
	};
	template <> struct conjunction<> : true_type {};

	template < typename T, typename... Ts>
	struct disjunction<T, Ts...> :
		bool_constant<T::value || disjunction<Ts...>::value>
	{
	};

	template <> struct disjunction<> : false_type {};

	template < typename From, typename To >
	using explicit_cast = decltype( static_cast<To>( declval<From>() ) );

	template < bool... Bs>
	constexpr bool require = conjunction<bool_constant<Bs>...>::value;

	template < bool... Bs>
	constexpr bool either = disjunction<bool_constant<Bs>...>::value;

	template < bool... Bs>
	constexpr bool disallow = not require<Bs...>;

	template < template < typename... > class Op, typename... Args >
	constexpr bool exists = is_detected<Op, Args...>::value;

	template < typename To, template < typename... > class Op, typename... Args >
	constexpr bool casts_to = exists<explicit_cast, detected_t<Op, Args...>, To>;

	template < typename To, template < typename...> class Op, typename... Args >
	constexpr bool converts_to = is_detected_convertible<To, Op, Args...>::value;

	template < typename Exact, template < typename...> class Op, typename... Args >
	constexpr bool identical_to = is_detected_exact<Exact, Op, Args...>::value;

	namespace ops
	{
		template < typename T, typename U > using equal_to = decltype( declval<T>() == declval<U>() );
		template < typename T, typename U > using less = decltype( declval<T>() < declval<U>() );
		template < typename T > using dereference = decltype( *declval<T>() );
		template < typename T > using arrow = decltype( declval<T>().operator->() );

		template < typename T > using postfix_increment = decltype( declval<T>()++ );
		template < typename T > using prefix_increment = decltype( ++declval<T>() );
	}

	namespace adl
	{
		using nv::swap;
		template <class T, class U = T>
		using swap_with = decltype( swap( declval<T>(), declval<U>() ) );
	}

	namespace alias
	{
		template < typename T > using value_type = typename T::value_type;
		template < typename T > using reference = typename T::reference;
		template < typename T > using pointer = typename T::pointer;

		template < typename T >
		using iterator_category = typename nv::iterator_traits<T>::iterator_category;
	} 

	namespace concepts
	{

		template < typename T, typename U >
		constexpr bool SwappableWith = exists<adl::swap_with, T, U>;

//		template < typename T >
//		constexpr bool CopyConstructible = is_copy_constructible<T>::value;

		template < typename T > constexpr bool CopyAssignable = is_copy_assignable<T>::value;
//		template < typename T > constexpr bool Destructible = is_destructible<T>::value;
		template < typename T > constexpr bool Swappable = SwappableWith<T&, T&>;
		template < typename T > constexpr bool Pointer = is_pointer<T>::value;

		template < typename T, typename U = T >
		constexpr bool EqualityComparable = converts_to<bool, ops::equal_to, T, U>;

		template < typename T >
		constexpr bool Iterator = require<
//			CopyConstructible<T>,
			CopyAssignable<T>,
//			Destructible<T>,
			Swappable<T>,
			exists<ops::postfix_increment, T>,
			exists<ops::prefix_increment, T>,
			exists<ops::dereference, T>
		>;

		template < typename T >
		constexpr bool InputIterator = either<
			Pointer<T>,
			require<
			EqualityComparable<T,T>,
			Iterator<T>,
			exists<alias::value_type, T>,
			exists<alias::reference, T>,
			either<
			identical_to<detected_t<alias::reference, T>, ops::dereference, T>,
			converts_to<detected_t<alias::value_type, T>, ops::dereference, T>
			>,
			identical_to<detected_t<alias::pointer, T>, ops::arrow, T>,
			converts_to<detected_t<alias::value_type, T>, ops::dereference, T&>,
			converts_to<input_iterator_tag, alias::iterator_category, T>
			>
		>;

	} /* namespace concepts */

#define NV_CONCEPT( TYPE, CONCEPT ) \
	static_assert( nv::concepts::CONCEPT<TYPE>, #TYPE " does not model " #CONCEPT );

}

#endif // NV_STL_TYPE_TRAITS_CONCEPTS_HH
