// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file function.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief function class
*/

#ifndef NV_STL_FUNCTIONAL_FUNCTION_HH
#define NV_STL_FUNCTIONAL_FUNCTION_HH

#include <nv/stl/functional/common.hh>
#include <nv/stl/functional/mem_fn.hh>
#include <nv/stl/functional/reference.hh>
#include <nv/stl/type_traits/common.hh>
#include <nv/stl/utility/common.hh>

namespace nv
{

	template < typename Signature >
	class function;

	namespace detail
	{
		template < typename T >
		struct cast_type_wrapper
		{
			cast_type_wrapper( T v ) : value( v ) {}
			T value;
		};

		// TODO : call conventions?
		template < typename F > 
		inline F& callable_functor( F& f ) { return f; }
		template < typename C, typename T > 
		inline detail::mem_fn_impl< T C::* > callable_functor( T C::* &f ) { return mem_fn(f); }
		template < typename C, typename T >
		inline detail::mem_fn_impl< T C::* const > callable_functor( T C::* const &f ) { return mem_fn( f ); }
		template < typename C, typename T > 
		inline detail::mem_fn_impl< T C::* volatile > callable_functor( T C::* volatile &f ) { return mem_fn(f); }
		template < typename C, typename T >
		inline detail::mem_fn_impl< T C::* const volatile > callable_functor( T C::* const volatile &f ) { return mem_fn( f ); }

		template < typename T >
		struct is_location_invariant 
			: is_trivially_copyable<T> {};

		template < typename T >
		struct is_location_invariant< cast_type_wrapper< T > >
			: is_location_invariant< T >{};

		class dummy_class;

#if NV_COMPILER == NV_MSVC 
#pragma warning( push )
#pragma warning( disable : 4121 ) 
#endif		
		union function_storage
		{
			void* dummy_object;
			const void* dummy_const_object;
			void* (*dummy_function_pointer)();
			void (dummy_class::*dummy_member_pointer)();
		};
#if NV_COMPILER == NV_MSVC 
#pragma warning( pop )
#endif	

		union any_data
		{
			void*       access()       { return &raw_data[0]; }
			const void* access() const { return &raw_data[0]; }

			template < typename T >
			T& access() { return *static_cast<T*>( access() ); }

			template < typename T >
			const T& access() const { return *static_cast<const T*>( access() ); }

			function_storage unused;
			char raw_data[sizeof( function_storage )];
		};

		template < typename Signature >
		constexpr bool function_not_empty( const function< Signature >& f )
		{
			return static_cast<bool>( f );
		}

		template < typename T >
		constexpr bool function_not_empty( const T*& f )
		{
			return f != 0;
		}

		template < typename C, typename T >
		constexpr bool function_not_empty( T C::* const& f )
		{
			return f != 0;
		}

		template < typename T >
		constexpr bool function_not_empty( const T& )
		{
			return true;
		}

		enum operation
		{
			op_clone_functor,
			op_destroy_functor,
		};

		template <
			typename F,
			bool local = is_location_invariant< F >::value && sizeof( F ) <= sizeof( function_storage )
		>
		struct function_storage_policy;

		template < typename F >
		struct function_storage_policy< F, true >
		{
			static F* get_pointer( const any_data& src )
			{
				const F* ptr = addressof( src.access<F>() );
				return const_cast<F*>( ptr );
			}

			static bool execute_op( any_data& dest, const any_data& src, operation op )
			{
				switch ( op )
				{
				case op_clone_functor:
					new ( dest.access() ) F( src.access<F>() );
					break;
				case op_destroy_functor:
					dest.access<F>().~F();
					break;
				default:
					break;
				}
				return false;
			}

			static void init_functor( any_data& fdata, F&& f )
			{
				new ( fdata.access() ) F( move( f ) );
			}

		};

		template < typename F >
		struct function_storage_policy< F, false >
		{
			static F* get_pointer( const any_data& src )
			{
				const F* ptr = src.access<F*>();
				return const_cast<F*>( ptr );
			}

			static bool execute_op( any_data& dest, const any_data& src, operation op )
			{
				switch ( op )
				{
				case op_clone_functor:
					dest.access<F*>() = new F( *src.access<F*>() );
					break;
				case op_destroy_functor:
					delete dest.access<F*>();
					break;
				default:
					break;
				}
				return false;
			}

			static void init_functor( any_data& fdata, F&& f )
			{
				fdata.access<F*>() = new F( move( f ) );
			}

		};

		template < typename F >
		struct function_reference_storage_policy : function_storage_policy< F* >
		{
			typedef function_storage_policy< F* > base_type;

			static void init_functor( any_data& fdata, reference_wrapper<F> f )
			{
				base_type::init_functor( fdata, addressof( f.get() ) );
			}

		};

		template < typename F >
		struct function_wrapped_storage_policy : function_storage_policy< cast_type_wrapper< F > >
		{
			typedef function_storage_policy< cast_type_wrapper< F > > base_type;

			static F* get_pointer( const any_data& src )
			{
				return &base_type::get_pointer( src )->value;
			}
		};


		template < typename Signature, typename Functor >
		class function_handler;

		template < typename R, typename F, typename... Args >
		class function_handler< R(Args...), F > : public function_storage_policy<F>
		{
			typedef function_storage_policy<F> base_type;
		public:
			static R invoke( const any_data& f, Args... args )
			{
				return ( *base_type::get_pointer( f ) )( forward<Args>( args )... );
			}
		};

		template < typename R, typename F, typename... Args >
		class function_handler< R( Args... ), reference_wrapper<F> > : public function_reference_storage_policy<F>
		{
			typedef function_reference_storage_policy<F> base_type;
		public:
			static R invoke( const any_data& f, Args... args )
			{
				return callable_functor( **base_type::get_pointer( f ) )( forward<Args>( args )... );
			}
		};

		template < typename Signature, typename Functor, bool IsMemFn = is_member_function_pointer<Functor>::value >
		class pick_function_handler;

		template < typename F, typename R, typename... Args >
		class pick_function_handler< R( Args... ), F, true > 
			: public function_wrapped_storage_policy< F >
		{
			typedef function_wrapped_storage_policy< F > base_type;
		public:
			static R invoke( const any_data& f, Args... args )
			{
				return mem_fn( *base_type::get_pointer( f ) )( forward<Args>( args )... );
			}
		};
		template < typename F, typename R, typename... Args >
		class pick_function_handler< R( Args... ), F, false >
			: public function_handler<  R( Args... ), F >
		{
		};


	}

	template < typename R, typename... Args >
	class function< R( Args... ) > 
	{
		typedef R signature_type( Args... );

		template < typename F >
		struct invoke_type
		{
			typedef decltype( ( detail::callable_functor( declval<F&>() ) ) ( declval<Args>()... ) ) type;
		};

		template < typename R1, typename R2 >
		struct is_call_convertible : is_convertible<R1, R2> {};

		template < typename R1 >
		struct is_call_convertible< R1, void > : true_type {};

		template < typename F >
		using is_callable = is_call_convertible< invoke_type<F>, R >;

		template < typename C, typename T >
		using requires = typename enable_if< C::value, T >::type;
	public:
		typedef R result_type;

		function() noexcept : m_manager( nullptr ), m_invoker( nullptr ) {}
		function( nullptr_t ) noexcept : m_manager( nullptr ), m_invoker( nullptr ) {}
		function( const function& f ) : m_manager( nullptr ), m_invoker( nullptr )
		{
			if ( static_cast<bool>( f ) )
			{
				m_invoker = f.m_invoker;
				m_manager = f.m_manager;
				f.m_manager( m_functor, f.m_functor, detail::op_clone_functor );
			}
		}
		function( function&& f ) : m_manager( nullptr ), m_invoker( nullptr ) { f.swap( *this ); }

		template < typename F, typename = requires<is_callable<F>, void > >
		function( F f ) : m_manager( nullptr ), m_invoker( nullptr )
		{
			typedef detail::pick_function_handler< signature_type, F > handler;

			if ( detail::function_not_empty( f ) )
			{
				handler::init_functor( m_functor, move( f ) );
				m_invoker = &handler::invoke;
				m_manager = &handler::execute_op;
			}
		}

		function& operator=( const function& f )
		{
			function( f ).swap( *this );
			return *this;
		}

		function& operator=( function&& f )
		{
			function( move( f ) ).swap( *this );
			return *this;
		}

		function& operator=( nullptr_t )
		{
			clear();
			return *this;
		}

		template < typename F >
		requires< is_callable<F>, function& > operator=( F&& f )
		{
			function( forward<F>( f ) ).swap( *this );
			return *this;
		}
		template < typename F >
		function& operator=( reference_wrapper<F> f ) noexcept
		{
			function( f ).swap( *this );
			return *this;
		}

		void swap( function& f )
		{
			nv::swap( m_functor, f.m_functor );
			nv::swap( m_manager, f.m_manager );
			nv::swap( m_invoker, f.m_invoker );
		}

		explicit operator bool() const noexcept
		{
			return !!m_manager;
		}

		R operator()( Args... args ) const
		{
			NV_ASSERT( m_invoker, "Empty function execution!" );
			return m_invoker( m_functor, forward<Args>( args )... );
		}

		~function()
		{
			clear();
		}

	private:

		void clear()
		{
			if ( m_manager ) m_manager( m_functor, m_functor, detail::op_destroy_functor );
			m_manager = nullptr;
			m_invoker = nullptr;
		}

		typedef R( *invoker_type )( const detail::any_data&, Args... );
		typedef bool( *manager_type )( detail::any_data&, const detail::any_data&, detail::operation );

		detail::any_data m_functor;
		manager_type     m_manager; 
		invoker_type     m_invoker;
	};



}

#endif // NV_STL_FUNCTIONAL_FUNCTION_HH

