// Copyright (C) 2012-2013 Kornel Kisielewicz
// This file is part of NV Libraries.
// For conditions of distribution and use, see copyright notice in nv.hh

#include "nv/gl/gl_program.hh"

#include "nv/gl/gl_enum.hh"
#include "nv/logging.hh"
#include "nv/lib/gl.hh"

#include <glm/glm.hpp>
#include <glm/gtc/type_ptr.hpp>

using namespace nv;

gl_shader::gl_shader( uint32 shader_type ) 
	: object_id(0), shader_type( shader_type )
{
	// no op
}

gl_shader::gl_shader( uint32 shader_type, const string& shader_code )
	: object_id(0), shader_type( shader_type )
{
	compile( shader_code );
}

gl_shader::~gl_shader()
{
	free();
}

bool gl_shader::compile( const string& shader_code )
{
	generate();

	const char* pc = shader_code.c_str();

	glShaderSource( object_id,   1, &pc, 0 );
	glCompileShader( object_id );

	return validate();
}

void gl_shader::generate()
{
	if ( is_valid() )
	{
		free();
	}
	object_id = glCreateShader( shader_type );
}

void gl_shader::free()
{
	glDeleteShader( object_id );
	object_id = 0;
}

bool gl_shader::validate()
{
	const uint32 buffer_size = 1024;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int compile_ok = GL_FALSE;
	glGetShaderiv(object_id, GL_COMPILE_STATUS, &compile_ok);
	glGetShaderInfoLog( object_id, buffer_size, &length, buffer );

	if ( length > 0 )
	{
		if ( compile_ok == 0 )
		{
			NV_LOG( LOG_ERROR, "Shader #" << object_id << " error: " << buffer );
		}
		else
		{
			NV_LOG( LOG_INFO, "Shader #" << object_id << " compiled successfully: " << buffer );
		}
	}
	else
	{
		NV_LOG( LOG_INFO, "Shader #" << object_id << " compiled successfully." );
	}
	return compile_ok != 0;
}

gl_program::gl_program( const string& vertex_program, const string& fragment_program )
	: vertex_shader( GL_VERTEX_SHADER ), fragment_shader( GL_FRAGMENT_SHADER )
{
	compile( vertex_program, fragment_program );
}

gl_program::~gl_program()
{
	if ( is_valid() )
	{
		// Detach the shaders from the program
		glDetachShader( m_name.get_value(), vertex_shader.get_id() );
		glDetachShader( m_name.get_value(), fragment_shader.get_id() );
	}
}

bool gl_program::compile( const string& vertex_program, const string& fragment_program )
{
	if (!vertex_shader.compile( vertex_program )) { return false; }
	if (!fragment_shader.compile( fragment_program )) { return false; }

	glAttachShader( m_name.get_value(), fragment_shader.get_id() );
	glAttachShader( m_name.get_value(), vertex_shader.get_id() );
	glLinkProgram( m_name.get_value() );

	if (!validate())
	{
		return false;
	}
	load_attributes();
	load_uniforms();
	return true;
}

void gl_program::bind()
{
	glUseProgram( m_name.get_value() );
	update_uniforms();
}

void gl_program::unbind()
{
	glUseProgram( 0 );
}

bool gl_program::is_valid() const
{
	return m_name.is_valid();
}

void gl_program::load_attributes()
{
	int params;
	glGetProgramiv( m_name.get_value(), GL_ACTIVE_ATTRIBUTES, &params );

	for ( int i = 0; i < params; ++i )
	{
		int attr_nlen;
		int attr_len;
		unsigned attr_type;
		char name_buffer[128];

		glGetActiveAttrib( m_name.get_value(), i, 128, &attr_nlen, &attr_len, &attr_type, name_buffer );

		string name( name_buffer, attr_nlen );

		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int attr_loc = glGetAttribLocation( m_name.get_value(), name.c_str() );

		m_attribute_map[ name ] = new attribute( name, attr_loc, gl_enum_to_type( attr_type ), attr_len );
	}
}

void gl_program::load_uniforms()
{
	int params;
	glGetProgramiv( m_name.get_value(), GL_ACTIVE_UNIFORMS, &params );

	for ( int i = 0; i < params; ++i )
	{
		int uni_nlen;
		int uni_len;
		unsigned uni_type;
		char name_buffer[128];

		glGetActiveUniform( m_name.get_value(), i, 128, &uni_nlen, &uni_len, &uni_type, name_buffer );

		string name( name_buffer, uni_nlen );

		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int uni_loc = glGetUniformLocation( m_name.get_value(), name.c_str() );
		etype utype = gl_enum_to_type( uni_type );
		m_uniform_map[ name ] = create_uniform( utype, name, uni_loc, uni_len );
	}
}

void gl_program::update_uniforms()
{
	for ( uniform_map::iterator i = m_uniform_map.begin(); 	i != m_uniform_map.end(); ++i ) 
	{
		uniform_base* ubase = i->second;
		if ( ubase->is_dirty() )
		{
			int uloc = ubase->get_location();
			switch( ubase->get_type() )
			{
			case FLOAT          : glUniform1f( uloc, ((uniform< enum_to_type< FLOAT >::type >*)( ubase ))->get_value() ); break;
			case INT            : glUniform1i( uloc, ((uniform< enum_to_type< INT >::type >*)( ubase ))->get_value() ); break;
			case FLOAT_VECTOR_2 : glUniform2fv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< FLOAT_VECTOR_2 >::type >*)( ubase ))->get_value()) ); break;
			case FLOAT_VECTOR_3 : glUniform3fv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< FLOAT_VECTOR_3 >::type >*)( ubase ))->get_value()) ); break;
			case FLOAT_VECTOR_4 : glUniform4fv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< FLOAT_VECTOR_4 >::type >*)( ubase ))->get_value()) ); break;
			case INT_VECTOR_2   : glUniform2iv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< INT_VECTOR_2 >::type >*)( ubase ))->get_value()) ); break;
			case INT_VECTOR_3   : glUniform3iv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< INT_VECTOR_3 >::type >*)( ubase ))->get_value()) ); break;
			case INT_VECTOR_4   : glUniform4iv( uloc, 1, glm::value_ptr(((uniform< enum_to_type< INT_VECTOR_4 >::type >*)( ubase ))->get_value()) ); break;
			case FLOAT_MATRIX_2 : glUniformMatrix2fv( uloc, 1, GL_FALSE, glm::value_ptr(((uniform< enum_to_type< FLOAT_MATRIX_2 >::type >*)( ubase ))->get_value()) ); break;
			case FLOAT_MATRIX_3 : glUniformMatrix3fv( uloc, 1, GL_FALSE, glm::value_ptr(((uniform< enum_to_type< FLOAT_MATRIX_3 >::type >*)( ubase ))->get_value()) ); break;
			case FLOAT_MATRIX_4 : glUniformMatrix4fv( uloc, 1, GL_FALSE, glm::value_ptr(((uniform< enum_to_type< FLOAT_MATRIX_4 >::type >*)( ubase ))->get_value()) ); break;
			//default     : error?
			}
			ubase->clean();
		}
	}

}


bool gl_program::validate()
{
	const uint32 buffer_size = 2048;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int status;

	glGetProgramiv( m_name.get_value(), GL_LINK_STATUS, &status );
	glGetProgramInfoLog( m_name.get_value(), buffer_size, &length, buffer );

	NV_LOG( LOG_INFO, "Program #" << m_name.get_value() << (status == GL_FALSE ? " failed to compile!" : " compiled successfully.") );

	if ( length > 0 )
	{
		NV_LOG( LOG_INFO, "Program #" << m_name.get_value() << " log: " << buffer );
	}

	if ( status == GL_FALSE ) 
	{
		return false;
	}

	glValidateProgram( m_name.get_value() );
	glGetProgramiv( m_name.get_value(), GL_VALIDATE_STATUS, &status );

	if ( status == GL_FALSE )
	{
		glGetProgramInfoLog( m_name.get_value(), buffer_size, &length, buffer );
		NV_LOG( LOG_ERROR, "Program #" << m_name.get_value() << " validation error : " << buffer );
		return false;
	}
	return true;
}


