// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file unique.hh
* @author Kornel Kisielewicz
* @brief unique_ptr implementation
*/

#ifndef NV_STL_MEMORY_UNIQUE_HH
#define NV_STL_MEMORY_UNIQUE_HH

#include <nv/stl/memory/common.hh>
#include <nv/stl/type_traits/common.hh>

namespace nv
{
	template < typename T >
	class unique_ptr
	{
	public:
		typedef T*               pointer;
		typedef T                element_type;

		unique_ptr() : ptr( pointer() ) {}
		explicit unique_ptr( pointer p ) : ptr( p ) {}

		unique_ptr( unique_ptr&& u ) : ptr( u.release() ) {}

		template< typename U >
		unique_ptr( unique_ptr<U> u ) : ptr( u.release() ) {}
		~unique_ptr() { reset(); }

		unique_ptr& operator=( unique_ptr&& u )
		{
			reset( u.release() );
			return *this;
		}

		template< typename U >
		unique_ptr& operator=( unique_ptr<U>&& u )
		{
			reset( u.release() );
			return *this;
		}

		typename add_lvalue_reference<element_type>::type 
		operator*() const
		{
			NV_ASSERT( ptr != nullptr, "dereferencing null ptr!" );
			return *ptr;
		}
		pointer operator->() const
		{
			NV_ASSERT( ptr != nullptr, "dereferencing null ptr!" );
			return ptr;
		}

		pointer get() const
		{
			return ptr;
		}

		pointer release()
		{
			pointer p = ptr;
			ptr = 0;
			return p;
		}

		void reset( pointer p = pointer() )
		{
			if ( p != ptr )
			{
				delete ptr;
				ptr = p;
			}
		}

		void swap( unique_ptr& u )
		{
			using nv::swap;
			swap( ptr, u.ptr );
		}

		unique_ptr( const unique_ptr& ) = delete;
		template< typename U >
		unique_ptr( const unique_ptr<U>& ) = delete;
		unique_ptr& operator=( const unique_ptr& ) = delete;
		template< typename U >
		unique_ptr& operator=( const unique_ptr<U>& ) = delete;
	private:
		pointer ptr;
	};

	template < typename T >
	class unique_ref
	{
	public:
		typedef T*               pointer;
		typedef T                element_type;

		unique_ref( unique_ref&& u ) : ptr( u.release() ) {}

		template< typename U >
		unique_ref( unique_ref<U> u ) : ptr( u.release() ) {}
		~unique_ref() { reset(); }

		unique_ref& operator=( unique_ref&& u )
		{
			reset( u.release() );
			return *this;
		}

		template< typename U >
		unique_ref& operator=( unique_ref<U>&& u )
		{
			reset( u.release() );
			return *this;
		}

		typename add_lvalue_reference<element_type>::type
			operator*() const
		{
			return *ptr;
		}
		pointer operator->() const
		{
			return ptr;
		}

	private:
		explicit unique_ref( pointer p ) : ptr( p ) {}

		pointer release()
		{
			pointer p = ptr;
			ptr = 0;
			return p;
		}

		void reset( pointer p = pointer() )
		{
			if ( p != ptr )
			{
				delete ptr;
				ptr = p;
			}
		}

	public:
		void swap( unique_ref& u )
		{
			using nv::swap;
			swap( ptr, u.ptr );
		}

		unique_ref( const unique_ref& ) = delete;
		template< typename U >
		unique_ref( const unique_ref<U>& ) = delete;
		unique_ref& operator=( const unique_ref& ) = delete;
		template< typename U >
		unique_ref& operator=( const unique_ref<U>& ) = delete;
	private:
		pointer ptr;
	};

}

#endif // NV_STL_MEMORY_UNIQUE_HH

