// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file filesystem.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief filesystem library
*/

#ifndef NV_STL_FILESYSTEM_HH
#define NV_STL_FILESYSTEM_HH

#include <nv/common.hh>
#include <nv/stl/utility.hh>

namespace nv
{

	enum class file_type
	{
		NOT_FOUND = -1,
		NONE      = 0,
		REGULAR   = 1,
		DIRECTORY = 2,
		SYMLINK   = 3,
		BLOCK     = 4,
		CHARACTER = 5,
		FIFO      = 6,
		SOCKET    = 7,
		UNKNOWN   = 8
	};

	enum class perms
	{
		NONE             = 0,
		OWNER_READ       = 0400, // S_IRUSR
		OWNER_WRITE      = 0200, // S_IWUSR
		OWNER_EXEC       = 0100, // S_IXUSR
		OWNER_ALL        = 0700, // S_IRWXU
		GROUP_READ       = 040,  // S_IRGRP
		GROUP_WRITE      = 020,  // S_IWGRP
		GROUP_EXEC       = 010,  // S_IXGRP
		GROUP_ALL        = 070,  // S_IRWXG
		OTHERS_READ      = 04,   // S_IROTH
		OTHERS_WRITE     = 02,   // S_IWOTH
		OTHERS_EXEC      = 01,   // S_IXOTH
		OTHERS_ALL       = 07,   // S_IRWXO
		ALL              = 0777,

		SET_UID          = 04000,// S_ISUID
		SET_GID          = 02000,// S_ISGID
		STICKY_BIT       = 01000,// S_ISVTX
		MASK             = 07777,

		UNKNOWN          = 0xFFFF,
		ADD_PERMS        = 0x10000,
		REMOVE_PERMS     = 0x20000,
		RESOLVE_SYMLINKS = 0x40000
	};

	inline perms& operator&=( perms& lhs, perms rhs ) { lhs = perms( static_cast<int>( lhs ) & static_cast<int>(rhs) ); return lhs; }
	inline perms& operator|=( perms& lhs, perms rhs ) { lhs = perms( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); return lhs; }
	inline perms& operator^=( perms& lhs, perms rhs ) { lhs = perms( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); return lhs; }
	inline constexpr perms operator&( perms lhs, perms rhs ) { return perms( static_cast<int>( lhs ) & static_cast<int>( rhs ) ); }
	inline constexpr perms operator|( perms lhs, perms rhs ) { return perms( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); }
	inline constexpr perms operator^( perms lhs, perms rhs ) { return perms( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); }
	inline constexpr perms operator~( perms lhs ) { return perms( ~static_cast<int>( lhs ) ); }

	enum class copy_options
	{
		NONE               = 0,
		SKIP_EXISTING      = 1,
		OVERWRITE_EXISTING = 2,
		UPDATE_EXISTING    = 4,
		RECURSIVE          = 8,
		COPY_SYMLINKS      = 16,
		SKIP_SYMLINKS      = 32,
		DIRECTORIES_ONLY   = 64,
		CREATE_SYMLINKS    = 128,
	};

	inline copy_options& operator&=( copy_options& lhs, copy_options rhs ) { lhs = copy_options( static_cast<int>( lhs ) & static_cast<int>( rhs ) ); return lhs; }
	inline copy_options& operator|=( copy_options& lhs, copy_options rhs ) { lhs = copy_options( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); return lhs; }
	inline copy_options& operator^=( copy_options& lhs, copy_options rhs ) { lhs = copy_options( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); return lhs; }
	inline constexpr copy_options operator&( copy_options lhs, copy_options rhs ) { return copy_options( static_cast<int>( lhs ) & static_cast<int>( rhs ) ); }
	inline constexpr copy_options operator|( copy_options lhs, copy_options rhs ) { return copy_options( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); }
	inline constexpr copy_options operator^( copy_options lhs, copy_options rhs ) { return copy_options( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); }
	inline constexpr copy_options operator~( copy_options lhs ) { return copy_options( ~static_cast<int>( lhs ) ); }

	enum class directory_options
	{
		NONE                     = 0,
		FOLLOW_DIRECTORY_SYMLINK = 1,
		SKIP_PERMISSION_DENIED   = 2,
	};

	inline directory_options& operator&=( directory_options& lhs, directory_options rhs ) { lhs = directory_options( static_cast<int>( lhs ) & static_cast<int>( rhs ) ); return lhs; }
	inline directory_options& operator|=( directory_options& lhs, directory_options rhs ) { lhs = directory_options( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); return lhs; }
	inline directory_options& operator^=( directory_options& lhs, directory_options rhs ) { lhs = directory_options( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); return lhs; }
	inline constexpr directory_options operator&( directory_options lhs, directory_options rhs ) { return directory_options( static_cast<int>( lhs ) & static_cast<int>( rhs ) ); }
	inline constexpr directory_options operator|( directory_options lhs, directory_options rhs ) { return directory_options( static_cast<int>( lhs ) | static_cast<int>( rhs ) ); }
	inline constexpr directory_options operator^( directory_options lhs, directory_options rhs ) { return directory_options( static_cast<int>( lhs ) ^ static_cast<int>( rhs ) ); }
	inline constexpr directory_options operator~( directory_options lhs ) { return directory_options( ~static_cast<int>( lhs ) ); }

	struct space_info
	{
		uintmax capacity;
		uintmax free;
		uintmax available;
	};

	class file_status
	{
	public:
		explicit file_status( file_type ftype = file_type::NONE, perms p = perms::UNKNOWN )
			: m_ftype( ftype ), m_perms( p )
		{
		}

		file_status( const file_status& ) = default;
		file_status& operator=( const file_status& ) = default;
		file_status( file_status&& ) = default;
		file_status& operator=( file_status&& ) = default;

		file_type type() const { return m_ftype;  }
		perms permissions() const { return m_perms; }
		void type( file_type ftype ) { m_ftype = ftype; }
		void permissions( perms p ) { m_perms = p; }
	private:
		file_type m_ftype;
		perms m_perms;
	};

}

#endif // NV_STL_FILESYSTEM_HH
