// Copyright (C) 2016-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file ecs.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_HH
#define NV_ECS_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/handle_manager.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/message_queue.hh>
#include <nv/ecs/component_storage.hh>


namespace nv
{

	namespace ecs
	{

		template < typename Handle, typename MessageList, typename Time = f32 >
		class ecs : public message_queue< MessageList, Time >
		{
		public:
			typedef message_queue< MessageList, Time > base_type;
			typedef ecs< Handle, MessageList, Time >   this_type;
			typedef Handle                             handle_type;
			typedef typename base_type::time_type      time_type;
			using typename base_type::message_list;
			using typename base_type::message_type;
			using typename base_type::message;

			using destructor_handler = function< void() >;
			using update_handler     = function< void( time_type ) >;
			using destroy_handler    = function< void( void* ) >;
			using create_handler     = function< void( handle_type, void*, const lua::stack_proxy& ) >;

			class component_interface
			{
			public:
				virtual void clear_() = 0;
				virtual void remove_( handle_type h ) = 0;
				virtual void on_attach( handle_type parent, handle_type child ) = 0;
				virtual void* get_raw( handle_type h ) = 0;
				virtual void* insert_raw( handle_type h ) = 0;
				virtual const void* get_raw( handle_type h ) const = 0;
				virtual uint32 temp_index( handle_type h ) const = 0;
				virtual void remove_by_index_( uint32 i ) = 0;


				void run_create( handle_type h, const lua::stack_proxy& p )
				{
					void* c = insert_raw( h );
					for ( auto cf : m_create )
						cf( h, c, p );
				}

				vector< create_handler >  m_create;
				vector< destroy_handler > m_destroy;

			};

			class enumerator : public value_enumerator_base < handle_type >
			{
			public:
				typedef value_enumerator_base < handle_type > base_class;
				explicit enumerator( const ecs& aecs, handle_type current = handle_type() ) : base_class( current ), m_ecs( aecs ) {}
				enumerator& operator++ ()
				{
					base_class::m_value = m_ecs.next( base_class::m_value );
					return *this;
				}
			private:
				const ecs& m_ecs;
			};

			template< typename Component >
			class component_enumerator : public iterator< input_iterator_tag, Component >
			{
			public:
				typedef value_enumerator_base < handle_type > base_class;
				explicit component_enumerator( ecs& aecs, handle_type current = handle_type() )
					: m_ecs( aecs ), m_handle( current ), m_component( nullptr )
				{
					m_interface = m_ecs.get_interface< Component >();
					if ( m_interface && m_handle )
					{
						m_component = (Component*)m_interface->get_raw( current );
						if ( !m_component )
							find_next();
					}
					else
						m_handle = handle_type();
				}
				component_enumerator& operator++ ()
				{
					find_next();
					return *this;
				}
				Component& operator* () { return *m_component; }
				const Component& operator* () const { return *m_component; }
				bool operator== ( const component_enumerator& rhs ) const
				{
					return m_handle == rhs.m_handle;
				}
				bool operator!= ( const component_enumerator& rhs ) const
				{
					return !( *this == rhs );
				}
			private:
				void find_next()
				{
					if ( !m_interface ) return;
					do
					{
						m_handle = m_ecs.next( m_handle );
						m_component = (Component*)m_interface->get_raw( m_handle );
					} while ( m_handle && !m_component );
				}
				component_interface* m_interface;
				handle_type m_handle;
				Component* m_component;
				ecs& m_ecs;
			};

			template < typename Component >
			handle_type handle_cast( const Component& c )
			{
				return *reinterpret_cast<const handle_type*>( &c );
			}

			template < int, typename... Components >
			struct gather_components
			{
				static constexpr uint32 SIZE = sizeof...( Components );
				component_interface* cis[SIZE];
				void* cmps[SIZE];

				gather_components( this_type& ecs )
				{
					fill< 0, Components... >( ecs );
				}

				bool run( handle_type h )
				{
					for ( uint32 i = 0; i < SIZE; ++i )
					{
						cmps[i] = cis[i]->get_raw( h );
						if ( !cmps[i] ) return false;
					}
					return true;
				}

				template< typename Component >
				bool runc( Component& c )
				{
					return run( *reinterpret_cast<const handle_type*>( &c ) );
				}

				template < typename SC >
				SC& get()
				{
					return run_get< 0, SC, Components...>();
				}

			private:
				template < int Index, typename C, typename... Cs >
				void fill( this_type& ecs )
				{
					cis[Index] = ecs.get_interface< C >();
					NV_ASSERT( cis[Index], "What the fuck is this?" );
					fill< Index + 1, Cs... >( ecs );
				}

				template < int Index >
				void fill( this_type& ) {}

				template < int Index, typename SC, typename C, typename... Cs >
				C& run_get()
				{
					return get_impl< Index, SC, C, Cs... >( nv::is_same< SC, C >{} );
				}

				template < int Index, typename SC, typename C, typename C2, typename... Cs >
				C& get_impl( false_type&& )
				{
					return get_impl< Index + 1, SC, C2, Cs...>( nv::is_same< SC, C >() );
				}

				template < int Index, typename SC, typename C, typename... Cs >
				C& get_impl( true_type&& )
				{
					return *(C*)cmps[Index];
				}

			};

			template <int I>
			struct gather_components<I>
			{
				gather_components( this_type& ) {}
				bool run( handle_type ) { return true;  }
				template < typename Component >
				bool runc( Component& ) { return true; }
			};

			template< typename System, typename... Args >
			System* register_system( string_view name, Args&&... args )
			{
				System* result = new System( nv::forward< Args >( args )... );

				this->template register_handler< System >( name, (System*)( result ) );
				register_component_helper< System >( result, typename has_components< System >::type() );
				register_ecs_update< System >( (System*)( result ),
					typename has_ecs_update< this_type, System, time_type >::type() );
				m_cleanup.emplace_back( [=] () { delete result; } );
				return result;
			}

			template< typename System >
			void register_component_helper( System* c, true_type&& )
			{
				using component_list = typename System::components;
				register_component_messages< System, component_list >( c, message_list() );
				register_ecs_component_update< System >( c, component_list(),
					typename has_ecs_component_update< this_type, System, component_list, time_type >::type() );
				register_component_update< System >( c, component_list(),
					typename has_component_update< System, component_list, time_type >::type() );
				register_destroy< System, mpl::head<component_list> >( (System*)( c ),
					typename has_destroy< System, mpl::head<component_list> >::type() );
			}

			template< typename System >
			void register_component_helper( System*, false_type&& )
			{
			}

			template < typename Component, typename Handler >
			component_storage* register_component( string_view name, component_interface* c )
			{
				auto s = new component_storage_handler< Component >;
				m_cstorage.push_back( s );
				m_cmap[thash64::create<Component>()] = s;
				m_cmap_by_name[name] = s;

				m_components.push_back( c );
				m_component_map[thash64::create<Component>()] = c;
				m_component_map_by_name[name] = c;
				this->template register_handler< Handler >( name, (Handler*)c );
				register_component_messages< Handler, mpl::list< Component > >( (Handler*)( c ), message_list() );
				register_ecs_component_update< Handler >( (Handler*)( c ), mpl::list< Component >(),
					typename has_ecs_component_update< this_type, Handler, mpl::list< Component >, time_type >::type() );
				register_component_update< Handler >( (Handler*)( c ), mpl::list< Component >(),
					typename has_component_update< Handler, mpl::list< Component >, time_type >::type() );
				register_ecs_update< Handler >( (Handler*)( c ), 
					typename has_ecs_update< this_type, Handler, time_type >::type() );
				register_destroy< Handler, Component >( (Handler*)( c ),
					typename has_destroy< Handler, Component >::type() );
				register_create< Handler, Component, handle_type >( (Handler*)( c ),
					typename has_create< Handler, Component, handle_type >::type() );
				
				return s;
			}

			template < typename ComponentManager, typename ...Args >
			ComponentManager* register_component_manager( Args&&... args )
			{
				ComponentManager* result = new ComponentManager( *this, nv::forward< Args >( args )... );
				m_cleanup.emplace_back( [=] () { delete result; } );
				return result;
			}

			handle_type create()
			{
				return m_handles.create_handle();
			}

			void update( time_type dtime )
			{
				this->update_time( dtime );
				for ( auto u : m_update_handlers )
					u( dtime );
				for ( auto h : m_dead_handles )
					remove( h );
				m_dead_handles.clear();
			}

			void clear()
			{
				this->reset_events();
				for ( auto c : m_components )
					c->clear_();
				m_handles.clear();
			}

			~ecs()
			{
				for ( auto s : m_cstorage )
					delete s;
				m_cstorage.clear();
				// 				for ( auto cs : m_component_storage )
				// 					delete cs;
				//				m_cstorage.clear();
				// reverse order delete;
				if ( !m_cleanup.empty() )
				for ( int i = (int)m_cleanup.size() - 1; i >= 0; --i )
					m_cleanup[i]();
				m_cleanup.clear();
			}

			virtual void attach( handle_type parent, handle_type child )
			{
				m_handles.detach( child );
				m_handles.attach( parent, child );
				for ( auto c : m_components )
					c->on_attach( parent, child );
			}

			template< typename Component >
			uint32 get_debug_index( handle_type h )
			{
				return get_interface<Component>()->temp_index( h );
			}

			handle_type get_parent( handle_type h ) const
			{
				return h ? m_handles.get_parent( h ) : handle_type();
			}

			handle_type next( handle_type h ) const
			{
				return h ? m_handles.next( h ) : handle_type();
			}

			handle_type first_child( handle_type h ) const
			{
				return h ? m_handles.first( h ) : handle_type();
			}

			bool is_valid( handle_type h ) const
			{
				return m_handles.is_valid( h );
			}

			enumerator_provider< enumerator > children( handle_type h ) const
			{
				return enumerator_provider< enumerator >(
					enumerator( *this, first_child( h ) ),
					enumerator( *this )
					);
			}

			template< typename Component >
			enumerator_provider< component_enumerator< Component > > child_components( handle_type h )
			{
				return enumerator_provider< component_enumerator< Component > >(
					component_enumerator< Component >( *this, first_child( h ) ),
					component_enumerator< Component >( *this )
					);
			}

			template < typename C, typename F >
			void remove_component_if( F&& f )
			{
				auto storage = get_storage<C>();
				auto temp_component = get_interface<C>();
				uint32 i = 0;
				while ( i < storage->size() )
					if ( f( ( *storage )[i] ) )
						temp_component->remove_by_index_( i );
					else
						++i;
			}

			template < typename C>
			void remove_component( handle_type h )
			{
				auto temp_component = get_interface<C>();
				temp_component->remove_( h );
			}

			template < typename C, typename F >
			void for_each( F&& f )
			{
				auto storage = get_storage<C>();
				NV_ASSERT( storage, "Invalid component" );
				for ( auto& c : *storage )
				{
					f( c );
				}
			}


			template < typename F >
			void recursive_call( handle_type h, F&& f )
			{
				f( h );
				for ( auto c : children( h ) )
					recursive_call( c, f );
			}

			void mark_remove( handle_type h )
			{
				m_dead_handles.push_back( h );
			}

			void remove( handle_type h )
			{
				handle_type ch = m_handles.first( h );
				while ( handle_type r = ch )
				{
					ch = m_handles.next( ch );
					remove( r );
				}
				for ( auto c : m_components )
					c->remove_( h );
				m_handles.free_handle( h );
			}

			bool exists( handle_type h ) const
			{
				return m_handles.is_valid( h );
			}

			component_interface* get_component( shash64 component_name )
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			const component_interface* get_component( shash64 component_name ) const
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			template < typename Component >
			component_interface* get_interface()
			{
				return m_component_map[thash64::create< Component >()];
			}

			template < typename Component >
			const component_interface* get_interface() const
			{
				return m_component_map[thash64::create< Component >()];
			}

			template < typename Component >
			Component* get( handle_type h )
			{
				auto it = m_component_map.find( thash64::create< Component >() );
				NV_ASSERT( it != m_component_map.end(), "Get fail!" );
				return static_cast<Component*>( it->second->get_raw( h ) );
			}

			template < typename Component >
			const Component* get( handle_type h ) const
			{
				auto it = m_component_map.find( thash64::create< Component >() );
				NV_ASSERT( it != m_component_map.end(), "Get fail!" );
				return static_cast<const Component*>( it->second->get_raw( h ) );
			}

			template < typename Component >
			component_storage_handler< Component >*
				get_storage()
			{
				return storage_cast<Component>( m_cmap[thash64::create< Component >()] );
			}

			template < typename Component >
			const component_storage_handler< Component >*
				get_storage() const
			{
				return storage_cast<Component>( m_cmap[thash64::create< Component >()] );
			}
			
			template < typename Component >
			Component& add_component( handle_type h )
			{
				component_interface* ci = get_interface<Component>();
				return *((Component*)ci->insert_raw( h ));
			}

			template < typename Component, typename ...Args >
			Component& add_component( handle_type h, Args&&... args )
			{
				component_interface* ci = get_interface<Component>();
				Component& result = *( (Component*)ci->insert_raw( h ) );
				result = Component{ nv::forward<Args>( args )... };
				return result;
			}

			template < typename System, typename Components, template <class...> class List, typename... Messages >
			void register_component_messages( System* h, List<Messages...>&& )
			{
				int unused_0[] = { ( register_ecs_component_message<System,Messages>( h, Components() , typename has_ecs_component_message< this_type, System, Components, Messages >::type() ), 0 )... };
				int unused_1[] = { ( register_component_message<System,Messages>( h, Components(), typename has_component_message< System, Components, Messages >::type() ), 0 )... };
				int unused_2[] = { ( register_ecs_message<System,Messages>( h, typename has_ecs_message< this_type, System, Messages >::type() ), 0 )... };
			}
			
			template < typename System, typename Message, typename C, typename... Cs >
			void register_component_message( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					const Message& m = message_cast<Message>( msg );
					auto callback = [=] ( handle_type h )
					{
						if ( void* c = ci->get_raw( h ) )
						{
							gather_components<0, Cs... > gather( *this );
							if ( gather.run( h ) )
								s->on( m, *( (C*)( c ) ), gather.template get<Cs>()... );
						}
					};
					if ( msg.recursive )
						this->recursive_call( m.entity, nv::move( callback ) );
					else
						callback( m.entity );
	
				} );
			}

			template < typename System, typename Message, typename C, typename... Cs >
			void register_ecs_component_message( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					const Message& m = message_cast<Message>( msg );
					auto callback = [=] ( handle_type h )
					{
						if ( void* c = ci->get_raw( h ) )
						{
							gather_components<0, Cs... > gather( *this );
							if ( gather.run( h ) )
								s->on( m, *this, *( (C*)( c ) ), gather.template get<Cs>()... );
						}
					};
					if ( msg.recursive )
						this->recursive_call( m.entity, callback );
					else
						callback( m.entity );
				} );

			}
			template < typename System, typename Message >
			void register_ecs_message( System* s, true_type&& )
			{
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					s->on( message_cast<Message>( msg ), *this );
				} );

			}

			template < typename System, typename Message, typename... Cs >
			void register_component_message( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename Message, typename... Cs >
			void register_ecs_component_message( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename Message >
			void register_ecs_message( System*, false_type&& ) {}

			template < typename System >
			void register_ecs_update( System* s, true_type&& )
			{
				register_update( [=] ( time_type dtime )
				{
					s->update( *this, dtime );
				} );
			}

			template < typename System, typename Component >
			void register_destroy( System* s, true_type&& )
			{
				component_interface* ci = get_interface<Component>();
				NV_ASSERT( ci, "Unregistered component!" );
				ci->m_destroy.push_back( [=] ( void* data )
				{
					s->destroy( *((Component*)data) );
				}
				);
			}

			template < typename System, typename Component, typename H >
			void register_create( System* s, true_type&& )
			{
				component_interface* ci = get_interface<Component>();
				NV_ASSERT( ci, "Unregistered component!" );
				ci->m_create.push_back( [=] ( H h, void* data, const lua::stack_proxy& p )
				{
					s->create( h, *( (Component*)data ), p );
				}
				);
			}

			template < typename System, typename C, typename... Cs >
			void register_component_update( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_storage_handler<C>* storage = get_storage<C>();
				register_update( [=] ( time_type dtime )
				{
					gather_components<0, Cs... > gather( *this );
					for ( auto& c : *storage )
					{
						if ( gather.runc( c ) )
							s->update( c, gather.template get<Cs>()..., dtime );
					}
				} );
			}

			template < typename System, typename C, typename... Cs >
			void register_ecs_component_update( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_storage_handler<C>* storage = get_storage<C>();
				register_update( [=] ( time_type dtime )
				{
					gather_components<0, Cs... > gather(*this);
					for ( auto& c : *storage )
					{
						if ( gather.runc( c ) )
							s->update( *this, c, gather.template get<Cs>()..., dtime );
					}
				} );
			}


			template < typename System >
			void register_ecs_update( System*, false_type&& ) {}

			template < typename System, typename Component >
			void register_destroy( System*, false_type&& ) {}

			template < typename System, typename Component, typename H >
			void register_create( System*, false_type&& ) {}

			template < typename System, typename... Cs >
			void register_component_update( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename... Cs >
			void register_ecs_component_update( System*, mpl::list< Cs...>&&, false_type&& ) {}

			void register_update( update_handler&& handler )
			{
				m_update_handlers.push_back( handler );
			}

		protected:

			handle_tree_manager< handle_type >          m_handles;
			vector< handle_type >                       m_dead_handles;
			vector< component_interface* >              m_components;
			hash_store< thash64, component_interface* > m_component_map;
			hash_store< shash64, component_interface* > m_component_map_by_name;
			vector< update_handler >                    m_update_handlers;

			vector< component_storage* >                m_cstorage;
			hash_store< thash64, component_storage* >   m_cmap;
			hash_store< shash64, component_storage* >   m_cmap_by_name;

			vector< component_interface* >              m_temp_components;
			vector< destructor_handler >                m_cleanup;
		};

	}




}

#endif // NV_ECS_HH
