// Copyright (C) 2016-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file ecs.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_HH
#define NV_ECS_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/handle_manager.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/message_queue.hh>
#include <nv/ecs/component_storage.hh>


namespace nv
{

	namespace lua
	{
		class stack_proxy;
	}

	namespace ecs
	{

		namespace detail
		{
			template<typename, typename T>
			struct has_update
			{
				static_assert( nv::integral_constant<T, false>::value, "Second template parameter needs to be of function type." );
			};

			template< typename C, typename Ret, typename... Args >
			struct has_update<C, Ret( Args... )>
			{
			private:
				template<typename T>
				static constexpr auto check( T* )
					-> typename nv::is_same< decltype( nv::declval<T>().update( nv::declval<Args>()... ) ), Ret >::type;

				template<typename>
				static constexpr nv::false_type check( ... );
			public:
				typedef decltype( check<C>( 0 ) ) type;
				static constexpr bool value = type::value;
			};

			template < typename S, typename T, typename Cs >
			struct has_ct_update_helper;

			template < typename S, typename T, typename... Cs >
			struct has_ct_update_helper< S, T, mpl::list< Cs... > >
			{
				using type = detail::has_update<S, void( Cs&..., T ) >;
			};

			template < typename S, typename E, typename T, typename Cs >
			struct has_ect_update_helper;

			template < typename S, typename E, typename T, typename... Cs >
			struct has_ect_update_helper< S, E, T, mpl::list< Cs... > >
			{
				using type = detail::has_update<S, void( E&, Cs&..., T ) >;
			};

			template < typename S, typename E, typename M, typename Cs >
			struct has_ec_message_helper;

			template < typename S, typename E, typename M, typename... Cs >
			struct has_ec_message_helper< S, E, M, mpl::list< Cs... > >
			{
				using type = detail::has_message<S, void( const M&, E&, Cs&... ) >;
			};

			template < typename S, typename M, typename Cs >
			struct has_c_message_helper;

			template < typename S, typename M, typename... Cs >
			struct has_c_message_helper< S, M, mpl::list< Cs... > >
			{
				using type = detail::has_message<S, void( const M&, Cs&... ) >;
			};

		}



		template < typename E, typename S, typename T >
		using has_ecs_update = detail::has_update<S, void( E&, T ) >;

		template < typename S, typename Cs, typename T >
		using has_component_update = typename detail::has_ct_update_helper<S, T, Cs >::type;

		template < typename E, typename S, typename Cs, typename T >
		using has_ecs_component_update = typename detail::has_ect_update_helper<S, E, T, Cs >::type;

		template < typename S, typename Cs, typename M >
		using has_component_message = typename detail::has_c_message_helper<S, M, Cs >::type;

		template < typename E, typename S, typename Cs, typename M >
		using has_ecs_component_message = typename detail::has_ec_message_helper<S, E, M, Cs >::type;

		template < typename E, typename S, typename M >
		using has_ecs_message = detail::has_message<S, void( const M&, E& ) >;

		template < typename Handle, typename MessageList, typename Time = f32 >
		class ecs : public message_queue< MessageList, Time >
		{
		public:
			typedef message_queue< MessageList, Time > base_type;
			typedef ecs< Handle, MessageList, Time >   this_type;
			typedef Handle                             handle_type;
			using base_type::time_type;
			using base_type::message_type;
			using base_type::message;

			using update_handler = function< void( time_type ) >;

			class component_interface
			{
			public:
				virtual void update( time_type dtime ) = 0;
				virtual void clear() = 0;
				virtual void initialize( handle_type, lua::stack_proxy& ) = 0;
				virtual void remove( handle_type h ) = 0;
				virtual void* get_raw( handle_type h ) = 0;
				virtual const void* get_raw( handle_type h ) const = 0;
				virtual component_storage* storage() = 0;
			};

			class enumerator : public value_enumerator_base < handle_type >
			{
			public:
				typedef value_enumerator_base < handle_type > base_class;
				explicit enumerator( const ecs& aecs, handle_type current = handle_type() ) : base_class( current ), m_ecs( aecs ) {}
				enumerator& operator++ ()
				{
					base_class::m_value = m_ecs.next( base_class::m_value );
					return *this;
				}
			private:
				const ecs& m_ecs;
			};

			template< typename Component >
			class component_enumerator : public iterator< input_iterator_tag, Component >
			{
			public:
				typedef value_enumerator_base < handle_type > base_class;
				explicit component_enumerator( ecs& aecs, handle_type current = handle_type() )
					: m_ecs( aecs ), m_handle( current ), m_component( nullptr )
				{
					m_interface = m_ecs.get_interface< Component >();
					if ( m_interface && m_handle )
					{
						m_component = (Component*)m_interface->get_raw( current );
						if ( !m_component )
							find_next();
					}
					else
						m_handle = handle_type();
				}
				component_enumerator& operator++ ()
				{
					find_next();
					return *this;
				}
				Component& operator* () { return *m_component; }
				const Component& operator* () const { return *m_component; }
				bool operator== ( const component_enumerator& rhs ) const
				{
					return m_handle == rhs.m_handle;
				}
				bool operator!= ( const component_enumerator& rhs ) const
				{
					return !( *this == rhs );
				}
			private:
				void find_next()
				{
					if ( !m_interface ) return;
					do
					{
						m_handle = m_ecs.next( m_handle );
						m_component = (Component*)m_interface->get_raw( m_handle );
					} while ( m_handle && !m_component );
				}
				component_interface* m_interface;
				handle_type m_handle;
				Component* m_component;
				ecs& m_ecs;
			};

			template < typename Component >
			handle_type handle_cast( const Component& c )
			{
				return *reinterpret_cast<const handle_type*>( &c );
			}

			template < typename... Components >
			struct gather_components
			{
				static constexpr uint32 SIZE = sizeof...( Components );
				component_interface* cis[SIZE];
				void* cmps[SIZE];

				gather_components( this_type& ecs )
				{
					fill< 0, Components... >( ecs );
				}

				bool run( handle_type h )
				{
					for ( uint32 i = 0; i < SIZE; ++i )
					{
						cmps[i] = cis[i]->get_raw( h );
						if ( !cmps[i] ) return false;
					}
					return true;
				}

				template< typename Component >
				bool runc( Component& c )
				{
					return run( *reinterpret_cast<const handle_type*>( &c ) );
				}

				template < typename SC >
				SC& get()
				{
					return run_get< 0, SC, Components...>();
				}

			private:
				template < int Index, typename C, typename... Components >
				void fill( this_type& ecs )
				{
					cis[Index] = ecs.get_interface< C >();
					NV_ASSERT( cis[Index], "What the fuck is this?" );
					fill< Index + 1, Components... >( ecs );
				}

				template < int Index >
				void fill( this_type& ) {}

				template < int Index, typename SC, typename C, typename... Components >
				C& run_get()
				{
					return get_impl< Index, SC, C, Components... >( nv::is_same< SC, C >{} );
				}

				template < int Index, typename SC, typename C, typename C2, typename... Components >
				C& get_impl( false_type&& )
				{
					return get_impl< Index + 1, SC, C2, Components...>( nv::is_same< SC, C >() );
				}

				template < int Index, typename SC, typename C, typename... Components >
				C& get_impl( true_type&& )
				{
					return *(C*)cmps[Index];
				}

			};

			template <>
			struct gather_components<>
			{
				gather_components( this_type& ) {};
				bool run( handle_type ) { return true;  }
				template < typename Component >
				bool runc( Component& ) { return true; }
			};

			template< typename System >
			void register_system( string_view name, System* c )
			{
				register_handler< System >( name, (System*)( c ) );
				register_ecs_update< System >( (System*)( c ),
					has_ecs_update< this_type, System, time_type >::type{} );
			}

			template < typename Component, typename Handler >
			void register_component( string_view name, component_interface* c )
			{
				m_components.push_back( c );
				m_component_map[thash64::create<Component>()] = c;
				m_component_map_by_name[name] = c;
				register_handler< Handler >( name, (Handler*)c );
				register_component_messages< Handler, mpl::list< Component > >( (Handler*)( c ), message_list{} );
				register_ecs_component_update< Handler >( (Handler*)( c ), mpl::list< Component >{},
					has_ecs_component_update< this_type, Handler, mpl::list< Component >, time_type >::type{} );
				register_component_update< Handler >( (Handler*)( c ), mpl::list< Component >{}, 
					has_component_update< Handler, mpl::list< Component >, time_type >::type{} );
				register_ecs_update< Handler >( (Handler*)( c ), 
					has_ecs_update< this_type, Handler, time_type >::type{} );
				//				auto s = c->storage();
				// 				m_cstorage.push_back( s );
				// 				m_cmap[thash64::create<Component>()] = s;
				// 				m_cmap_by_name[name] = s;
			}

			handle_type create()
			{
				return m_handles.create_handle();
			}

			void update( time_type dtime )
			{
				update_time( dtime );
				for ( auto c : m_components )
					c->update( dtime );
				for ( auto u : m_update_handlers )
					u( dtime );
			}

			void clear()
			{
				reset_events();
				for ( auto c : m_components )
					c->clear();
				m_handles.clear();
			}

			~ecs()
			{
				// 				for ( auto cs : m_component_storage )
				// 					delete cs;
				//				m_cstorage.clear();

			}

			virtual void attach( handle_type parent, handle_type child )
			{
				m_handles.detach( child );
				m_handles.attach( parent, child );
			}

			handle_type get_parent( handle_type h ) const
			{
				return h ? m_handles.get_parent( h ) : handle_type();
			}

			handle_type next( handle_type h ) const
			{
				return h ? m_handles.next( h ) : handle_type();
			}

			handle_type first_child( handle_type h ) const
			{
				return h ? m_handles.first( h ) : handle_type();
			}

			enumerator_provider< enumerator > children( handle_type h ) const
			{
				return enumerator_provider< enumerator >(
					enumerator( *this, first_child( h ) ),
					enumerator( *this )
					);
			}

			template< typename Component >
			enumerator_provider< component_enumerator< Component > > child_components( handle_type h )
			{
				return enumerator_provider< component_enumerator< Component > >(
					component_enumerator< Component >( *this, first_child( h ) ),
					component_enumerator< Component >( *this )
					);
			}

			template < typename F >
			void recursive_call( handle_type h, F&& f )
			{
				f( h );
				for ( auto c : children( h ) )
					recursive_call( c, f );
			}

			void remove( handle_type h )
			{
				handle_type ch = m_handles.first( h );
				while ( handle_type r = ch )
				{
					ch = m_handles.next( ch );
					remove( r );
				}
				for ( auto c : m_components )
					c->remove( h );
				m_handles.free_handle( h );
			}

			bool exists( handle_type h ) const
			{
				return m_handles.is_valid( h );
			}

			component_interface* get_component( shash64 component_name )
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			const component_interface* get_component( shash64 component_name ) const
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			template < typename Component >
			component_interface* get_interface()
			{
				return m_component_map[thash64::create< Component >()];
			}

			template < typename Component >
			const component_interface* get_interface() const
			{
				return m_component_map[thash64::create< Component >()];
			}

			template < typename Component >
			Component* get( handle_type h )
			{
				return static_cast<Component*>( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

			template < typename Component >
			const Component* get( handle_type h ) const
			{
				return static_cast<const Component*>( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

			template < typename Component >
			typename component_storage_handler< Component >*
				get_storage()
			{
				return storage_cast<Component>( m_cmap[thash64::create< Component >()] );
			}

			template < typename Component >
			const typename component_storage_handler< Component >*
				get_storage() const
			{
				return storage_cast<Component>( m_cmap[thash64::create< Component >()] );
			}

			template < typename System, typename Components, template <class...> class List, typename... Messages >
			void register_component_messages( System* h, List<Messages...>&& )
			{
				int unused_0[] = { ( register_ecs_component_message<System,Messages>( h, Components{} , has_ecs_component_message< this_type, System, Components, Messages >::type{} ), 0 )... };
				int unused_1[] = { ( register_component_message<System,Messages>( h, Components{}, has_component_message< System, Components, Messages >::type{} ), 0 )... };
				int unused_2[] = { ( register_ecs_message<System,Messages>( h, has_ecs_message< this_type, System, Messages >::type{} ), 0 )... };
			}

			template < typename System, typename Message, typename C, typename... Cs >
			void register_component_message( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					const Message& m = message_cast<Message>( msg );
					auto callback = [=] ( handle_type h )
					{
						if ( void* c = ci->get_raw( h ) )
						{
							gather_components< Cs... > gather( *this );
							if ( gather.run( h ) )
								s->on( m, *( (C*)( c ) ), gather.get<Cs>()... );
						}
					};
					if ( msg.recursive )
						this->recursive_call( m.entity, nv::move( callback ) );
					else
						callback( m.entity );
	
				} );
			}

			template < typename System, typename Message, typename C, typename... Cs >
			void register_ecs_component_message( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					const Message& m = message_cast<Message>( msg );
					auto callback = [=] ( handle_type h )
					{
						if ( void* c = ci->get_raw( h ) )
						{
							gather_components< Cs... > gather( *this );
							if ( gather.run( h ) )
								s->on( m, *this, *( (C*)( c ) ), gather.get<Cs>()... );
						}
					};
					if ( msg.recursive )
						this->recursive_call( m.entity, callback );
					else
						callback( m.entity );
				} );

			}
			template < typename System, typename Message >
			void register_ecs_message( System* s, true_type&& )
			{
				register_callback( Message::message_id, [=] ( const message& msg )
				{
					s->on( message_cast<Message>( msg ), *this );
				} );

			}

			template < typename System, typename Message, typename... Cs >
			void register_component_message( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename Message, typename... Cs >
			void register_ecs_component_message( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename Message >
			void register_ecs_message( System*, false_type&& ) {}

			template < typename System >
			void register_ecs_update( System* s, true_type&& )
			{
				register_update( [=] ( time_type dtime )
				{
					s->update( *this, dtime );
				} );
			}

			template < typename System, typename C, typename... Cs >
			void register_component_update( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				component_storage_handler<C>* storage = storage_cast<C>( ci->storage() );
				register_update( [=] ( time_type dtime )
				{
					gather_components< Cs... > gather( *this );
					for ( auto& c : *storage )
					{
						if ( gather.runc( c ) )
							s->update( c, gather.get<Cs>()..., dtime );
					}
				} );
			}

			template < typename System, typename C, typename... Cs >
			void register_ecs_component_update( System* s, mpl::list< C, Cs...>&&, true_type&& )
			{
				component_interface* ci = get_interface<C>();
				component_storage_handler<C>* storage = storage_cast<C>( ci->storage() );
				register_update( [=] ( time_type dtime )
				{
					gather_components< Cs... > gather(*this);
					for ( auto& c : *storage )
					{
						if ( gather.runc( c ) )
							s->update( *this, c, gather.get<Cs>()..., dtime );
					}
				} );
			}


			template < typename System >
			void register_ecs_update( System*, false_type&& ) {}

			template < typename System, typename... Cs >
			void register_component_update( System*, mpl::list< Cs...>&&, false_type&& ) {}

			template < typename System, typename... Cs >
			void register_ecs_component_update( System*, mpl::list< Cs...>&&, false_type&& ) {}

			void register_update( update_handler&& handler )
			{
				m_update_handlers.push_back( handler );
			}

		protected:

			handle_tree_manager< handle_type >          m_handles;
			vector< component_interface* >              m_components;
			hash_store< thash64, component_interface* > m_component_map;
			hash_store< shash64, component_interface* > m_component_map_by_name;
			vector< update_handler >                    m_update_handlers;

			// 			vector< component_storage* >                m_cstorage;
			// 			hash_store< thash64, component_storage* >   m_cmap;
			// 			hash_store< shash64, component_storage* >   m_cmap_by_name;
		};

	}




}

#endif // NV_ECS_HH
