// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file component_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_COMPONENT_HH
#define NV_ECS_COMPONENT_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/component_storage.hh>

namespace nv
{
	namespace lua
	{
		class stack_proxy;
	}

	namespace ecs
	{

		template < 
			typename Ecs,
			typename Component,
			typename Handler,
			template < typename, typename > class IndexTable = index_table
		>
		class component : public Ecs::component_interface
		{
		public:
			typedef Ecs                                     ecs_type;
			typedef typename ecs_type::message              message_type;
			typedef typename ecs_type::handle_type          handle_type;
			typedef typename ecs_type::time_type            time_type;
			typedef IndexTable< handle_type, sint32 >       index_table_type;
			typedef Component                               value_type;
			typedef Component                               component_type;
			typedef component_storage_handler< value_type > storage_type;
//			typedef vector< value_type > storage_type;
			typedef typename index_table_type::index_type   index_type;

			typedef typename storage_type::iterator         iterator;
			typedef typename storage_type::const_iterator   const_iterator;
			typedef typename storage_type::reference        reference;
			typedef typename storage_type::const_reference  const_reference;

			component( ecs_type& a_ecs, string_view a_name, bool relational = false, uint32 reserve = 0 ) 
				: m_ecs( a_ecs ), m_relational( relational )
			{
				m_ecs.register_component<component_type, Handler>( a_name, this );

				if ( reserve != 0 )
				{
					m_data.reserve( reserve );
				}
			}

			virtual void initialize( handle_type, lua::stack_proxy& ) {};

			inline value_type& insert( handle_type h )
			{
				index_type i = m_index.insert( h );
				NV_ASSERT( i == index_type( m_data.size() ), "Fail!" );
				NV_UNUSED( i );
				m_data.emplace_back();
				return m_data.back();
			}

			template < typename ...Args >
			value_type& insert( handle_type h, Args&&... args )
			{
				index_type i = m_index.insert( h );
				NV_ASSERT( i == index_type( m_data.size() ), "Fail!" );
				NV_UNUSED( i );
				m_data.emplace_back( nv::forward<Args>( args )... );
				return m_data.back();
			}

			bool exists( handle_type h )
			{
				return m_index.exists( h );
			}

			virtual void update( time_type /*dtime*/ )
			{
				// no-op
			}

			virtual void clear()
			{
 				for ( uint32 i = 0; i < m_data.size(); ++i )
 					destroy( &m_data[i] );
				m_index.clear();
 				m_data.clear();
			}

			value_type* get( handle_type h )
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}
			const value_type* get( handle_type h ) const
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			void* get_raw( handle_type h ) { return get( h ); }
			const void* get_raw( handle_type h ) const { return get( h ); }

			virtual void remove( handle_type h ) override
			{
				value_type* v = get( h );
				if ( v == nullptr ) return;
				destroy( v );
				index_type dead_eindex = m_index.remove_swap( h );
				if ( dead_eindex == -1 ) return;
				if ( dead_eindex != static_cast<index_type>( m_data.size() - 1 ) )
				{
					m_data[unsigned( dead_eindex )] = move( m_data.back() );
				}
				m_data.pop_back();
				if ( m_relational )
					relational_rebuild( dead_eindex );
			}

			void remove_by_index( index_type i )
			{
				if ( i > size() ) return;
				destroy( &m_data[i] );
				index_type dead_eindex = m_index.remove_swap_by_index( i );
				if ( dead_eindex == -1 ) return;
				if ( dead_eindex != static_cast<index_type>( m_data.size() - 1 ) )
				{
					m_data[unsigned( dead_eindex )] = move( m_data.back() );
				}
				m_data.pop_back();
				if ( m_relational )
					relational_rebuild( dead_eindex );
			}

			template < typename F >
			void remove_if( F f )
			{
				uint32 i = 0;
				while ( i < size() )
				{
					auto& md = m_data[i];
					if ( f( md ) )
						remove_by_index( i );
					else
						++i;
				}
			}

			virtual void destroy( value_type* )
			{
				// cleanup
			}

			~component()
			{
				clear();
			}
			
			inline const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			inline value_type& operator[] ( index_type i ) { return m_data[i]; }

			inline size_t size() const { return m_data.size(); }

			inline iterator        begin() { return m_data.begin(); }
			inline const_iterator  begin()  const { return m_data.begin(); }

			inline iterator        end() { return m_data.end(); }
			inline const_iterator  end()  const { return m_data.end(); }

			inline virtual component_storage* storage() { return &m_data; }
		protected:
			ecs_type&        m_ecs;
			storage_type     m_data;
		private:
			void swap( handle_type a, handle_type b )
			{
				index_type ai = m_index.get( a );
				index_type bi = m_index.get( b );
				m_index.swap( a, b );
				value_type t = nv::move( m_data[ai] );
				m_data[ai] = nv::move( m_data[bi] );
				m_data[bi] = nv::move( t );
			}

			handle_type extract_owner( index_type i )
			{
				return *(handle_type*)( &( m_data[i] ) );
			}

			void relational_remove( handle_type a )
			{
				index_type i = m_index.get( a );
				remove( a );
				relational_rebuild( i );
			}

			void relational_rebuild( index_type i )
			{
				handle_type h = extract_owner( i );
				handle_type p = m_ecs.get_parent( h );
				if ( !p ) return;
				if ( i < m_index.get( p ) )
				{
					swap( h, p );
					relational_rebuild( i );
				}
			}
			bool             m_relational;
			index_table_type m_index;
		};

	}

}

#endif // NV_ECS_COMPONENT_HH
