// Copyright (C) 2016-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file ecs.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_HH
#define NV_ECS_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/handle_manager.hh>
#include <nv/stl/range.hh>
#include <nv/core/types.hh>
#include <nv/ecs/message_queue.hh>

namespace nv
{

	namespace lua
	{
		class stack_proxy;
	}

	namespace ecs
	{

		template < typename Handle, typename Time = f32 >
		class ecs : public message_queue< Handle, Time >
		{
		public:
			typedef message_queue< Handle, Time > base_class;
			using base_class::time_type;
			using base_class::handle_type;
			using base_class::message_type;
			using base_class::message;
			using base_class::receiver_interface;

			class component_interface : public receiver_interface
			{
			public:
				virtual void initialize( handle_type, lua::stack_proxy& ) = 0;
				virtual void remove( handle_type h ) = 0;
				virtual void* get_raw( handle_type h ) = 0;
				virtual const void* get_raw( handle_type h ) const = 0;
			};

			class enumerator : public value_enumerator_base < handle_type >
			{
			public:
				typedef value_enumerator_base < handle_type > base_class;
				explicit enumerator( const ecs& aecs, handle_type current = handle_type() ) : base_class( current ), m_ecs( aecs ) {}
				enumerator& operator++ ()
				{
					base_class::m_value = m_ecs.next( base_class::m_value );
					return *this;
				}
				value_enumerator_base operator++ ( int )
				{
					auto result = *this;
					base_class::m_value = m_ecs.next( base_class::m_value );
					return result;
				}
			private:
				const ecs& m_ecs;
			};

			template < typename Component >
			void register_component( string_view name, component_interface* c )
			{
				m_components.push_back( c );
				register_receiver( name, c );
				m_component_map[thash64::create<Component>()] = c;
				m_component_map_by_name[name] = c;
			}

			handle_type create()
			{
				return m_handles.create_handle();
			}

			void clear()
			{
				reset_events();
				for ( auto c : m_receivers )
					c->clear();
				m_handles.clear();
			}

			virtual void attach( handle_type parent, handle_type child )
			{
				m_handles.detach( child );
				m_handles.attach( parent, child );
			}

			handle_type get_parent( handle_type h ) const
			{
				return h ? m_handles.get_parent( h ) : handle_type();
			}

			handle_type next( handle_type h ) const
			{
				return h ? m_handles.next( h ) : handle_type();
			}

			handle_type first_child( handle_type h ) const
			{
				return h ? m_handles.first( h ) : handle_type();
			}

			enumerator_provider< enumerator > children( handle_type h ) const
			{
				return enumerator_provider< enumerator >(
					enumerator( *this, first_child( h ) ),
					enumerator( *this )
				);
			}

			void remove( handle_type h )
			{
				handle_type ch = m_handles.first( h );
				while ( handle_type r = ch )
				{
					ch = m_handles.next( ch );
					remove( r );
				}
				for ( auto c : m_components )
					c->remove( h );
				m_handles.free_handle( h );
			}

			bool exists( handle_type h ) const
			{
				return m_handles.is_valid( h );
			}

			component_interface* get_component( shash64 component_name )
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			const component_interface* get_component( shash64 component_name ) const
			{
				auto it = m_component_map_by_name.find( component_name );
				return it != m_component_map_by_name.end() ? it->second : nullptr;
			}

			template < typename Component >
			Component* get( handle_type h )
			{
				return static_cast< Component* >( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

			template < typename Component >
			const Component* get( handle_type h ) const
			{
				return static_cast<const Component*>( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

		protected:

			handle_tree_manager< handle_type >          m_handles;
			vector< component_interface* >              m_components;
			hash_store< thash64, component_interface* > m_component_map;
			hash_store< shash64, component_interface* > m_component_map_by_name;
		};

	}




}

#endif // NV_ECS_HH
