// Copyright (C) 2012-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.
//
// TODO: decouple from the type database
// TODO: paths for function registration?


#ifndef NV_LUA_STATE_HH
#define NV_LUA_STATE_HH

#include <nv/common.hh>
#include <nv/core/types.hh>
#include <nv/stl/flags.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/string_table.hh>
#include <nv/stl/type_traits/function.hh>

#include <nv/lua/lua_proxy.hh>
#include <nv/lua/lua_iterator.hh>
#include <nv/lua/lua_handle.hh>
#include <nv/lua/lua_path.hh>
#include <nv/lua/lua_values.hh>
#include <nv/lua/lua_dispatch.hh>


#ifdef NV_DEBUG
#define NV_LUA_STACK_ASSERT( state, value ) nv::lua::stack_assert __lua_stack_assert( (state), (value) );
#else
#define NV_LUA_STACK_ASSERT( (state), (value) )
#endif

namespace nv
{
	namespace lua
	{

		class state;
		class lua_iterator_provider;

		using lua_rtti_read_function  = bool(*)( state*, const type_entry*, void*, int index );
		using lua_rtti_push_function  = void(*)( state*, const type_entry*, void* );
		

		class type_data
		{
			class entry
			{

			};
		public:
			explicit type_data( type_database* db ) : m_type_database( db )
			{
				register_standard_types();
			}
			const type_database* get_type_database() const { return m_type_database; }

			template < typename T >
			void insert( lua_rtti_push_function p, lua_rtti_read_function r )
			{
				insert( thash64::create<T>(), p, r );
			}
			void insert( thash64 tid, lua_rtti_push_function p, lua_rtti_read_function r );

			template < typename T >
			lua_rtti_read_function* get_push() const
			{
				return get_read( thash64::create<T>() );
			}

			template < typename T >
			lua_rtti_push_function* get_push() const
			{
				return get_push( thash64::create<T>() );
			}

			const lua_rtti_read_function* get_read( thash64 h ) const
			{
				auto f = m_type_read.find( h );
				return f != m_type_read.end() ? &f->second : nullptr;
			}
			const lua_rtti_push_function* get_push( thash64 h ) const
			{
				auto f = m_type_push.find( h );
				return f != m_type_push.end() ? &f->second : nullptr;
			}
		protected:
			void register_standard_types();

			hash_store< thash64, lua_rtti_read_function >  m_type_read;
			hash_store< thash64, lua_rtti_push_function >  m_type_push;
			type_database*                                 m_type_database;
		};

		const int ret_multi = -1;

		class state_wrapper
		{
		public:
			state_wrapper( lua_State* a_state, type_data* types, bool a_owner ) : m_state( a_state ), m_lua_types( types ), m_owner( a_owner ), m_global( true ) {}
			virtual ~state_wrapper();

			template < typename R >
			R call( const path& p )
			{
				if ( push_function( p, m_global ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename... Args >
			R call(const path& p, Args&&... args )
			{
				if (push_function(p, m_global))
				{
					detail::push_values(m_state, forward<Args>(args)...);
					if (call_function(sizeof...(Args), detail::return_count< R >::value) == 0)
					{
						return detail::pop_return_value<R>(m_state);
					}
				}
				return R();
			}

			template< typename R >
			R get( const path& p )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state );
				}
				return R();
			}

			template< typename R, typename T >
			R get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R, typename T >
			R raw_get( const T& key )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, key );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state );
			}

			template< typename R >
			R get( const path& p, const R& def )
			{
				if ( p.resolve( m_state, m_global ) )
				{
					return detail::pop_return_value<R>( m_state, def );
				}
				return R();
			}

			template< typename R, typename T >
			R get( T&& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, ::nv::forward<T>( key ) );
				call_get();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}

			template< typename R, typename T >
			R raw_get( T&& key, const R& def )
			{
				if ( m_global ) push_global_table();
				detail::push_value( m_state, ::nv::forward<T>( key ) );
				call_get_raw();
				if ( m_global ) pop_global_table();
				return detail::pop_return_value<R>( m_state, def );
			}
			bool is_defined( const path& p ) { return is_defined( p, m_global ); }
			void register_native_function( lfunction f, string_view name );

			template < typename F, F f >
			void register_function( string_view name )
			{
				register_native_function( detail::function_wrapper< F, f >, name );
			}

			template < typename C, typename F, F f >
			void register_function( string_view name )
			{
				register_native_function( detail::object_method_wrapper< C, F, f >, name );
			}

			const type_data* get_type_data() const { return m_lua_types; }




		protected:
			bool is_defined( const path& p, bool global );
			bool push_function( const path& p, bool global );
			int call_function( int nargs, int nresults );
		private:
			void push_global_table();
			void pop_global_table();
			void call_get();
			void call_get_raw();
		protected:
			lua_State* m_state;
			type_data* m_lua_types;
			bool m_owner;
			bool m_global;
		};

		class stack_guard
		{
		public:
			stack_guard( state* aL );
			stack_guard( state& aL );
			int get_level() const { return m_level; }
			~stack_guard();
		private:
			state* L;
			int m_level;
		};

		class stack_assert
		{
		public:
			stack_assert( state* aL, int expected );
			stack_assert( state& aL, int expected );
			stack_assert( lua_State* sL, int expected );
			~stack_assert();
		private:
			lua_State* L;
			int m_expected;
		};


		class table_guard;
		class state : public state_wrapper
		{
			friend class stack_guard;
			friend class table_guard;

		public:
			explicit state( bool load_libs = false, type_database* types = nullptr );
			explicit state( lua_State* state, type_database* types = nullptr );
			bool do_string( string_view code, string_view name, int rvalues = 0 );
			bool do_file( string_view filename );
			int get_stack_size();
			void log_stack();

			lua_State* get_raw();
			ref register_object( void* o, string_view lua_name );
			ref register_proto( string_view id, string_view storage );
			void store_metadata( ref object_index, string_view metaname, void* pointer );
			void unregister_object( ref object_index );

			template <typename E>
			void register_enum()
			{
				register_enum( m_lua_types->get_type_database()->get_type<E>() );
			}
			void register_enum( const type_entry* type );
			void register_enum( string_view name, int value );
			void register_singleton( string_view name, void* o );

			void register_native_object_method( string_view lua_name, string_view name, lfunction f );
			template < typename F, F f >
			void register_object_method( string_view lua_name, string_view name )
			{
				register_native_object_method( lua_name, name, detail::object_method_wrapper< typename memfn_class_type<F>::type, F, f > );
			}
			operator lua_State*() { return m_state; }

			template < typename H >
			void register_handle( const H& handle )
			{
				nv::lua::register_handle( m_state, handle, true );
			}
			template < typename H >
			void unregister_handle( const H& handle )
			{
				nv::lua::unregister_handle( m_state, handle );
			}
			template < typename H >
			ref register_handle_component( const H& handle, string_view id )
			{
				nv::lua::push_handle( m_state, handle );
				return register_handle_component_impl( id, true );
			}
			template < typename H >
			ref register_handle_component( const H& handle, string_view id, const path& proto )
			{
				if ( !proto.resolve( m_state, true ) )
				{
					return ref();
				}
				nv::lua::push_handle( m_state, handle );
				return register_handle_component_impl( id, false );
			}
			template < typename H >
			void unregister_handle_component( const H& handle, string_view id )
			{
				nv::lua::push_handle( m_state, handle );
				unregister_handle_component_impl( id );
			}

			using state_wrapper::call;

			template < typename R >
			R call_table( ref table, const path& p )
			{
				stack_guard guard( this ); // TODO: remove
				detail::push_ref_object( m_state,table );
				if ( push_function( p, false ) )
				{
					if ( call_function( 0, detail::return_count< R >::value ) == 0 )
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}
			template < typename R, typename... Args >
			R call_table( ref table, const path& p, Args&&... args )
			{
				stack_guard guard( this ); // TODO: remove
				detail::push_ref_object( m_state,table );
				if ( push_function( p, false ) )
				{
					detail::push_values( m_state, forward<Args>(args)... );
					if (call_function(sizeof...(Args), detail::return_count< R >::value) == 0)
					{
						return detail::pop_return_value<R>( m_state );
					}
				}
				return R();
			}

			virtual ~state();

			template < typename T >
			void register_rtti_type( lua_rtti_push_function p, lua_rtti_read_function r )
			{
				register_rtti_type( thash64::create< T >(), p, r );
			}

		private:
			void register_rtti_type( thash64 tid, lua_rtti_push_function p, lua_rtti_read_function r );
			ref register_handle_component_impl( string_view id, bool empty );
			void unregister_handle_component_impl( string_view id );

			int load_string( string_view code, string_view name );
			int load_file( string_view filename );
			int do_current( string_view name, int rvalues = 0 );
			void deep_pointer_copy( int index, void* obj );
		};




		class table_guard : public state_wrapper
		{
		public:
			table_guard( stack_proxy& proxy );
			table_guard( state* lstate, const path& p, bool global = true );
			table_guard( const table_guard& parent, const path& p );
			virtual ~table_guard();
			uint32 size();
			iterator_provider< kv_iterator > pairs()
			{
				return iterator_provider< kv_iterator >( m_parent, m_index );
			}
			iterator_provider< key_iterator > keys()
			{
				return iterator_provider< key_iterator >( m_parent, m_index );
			}
			iterator_provider< value_iterator > values()
			{
				return iterator_provider< value_iterator >( m_parent, m_index );
			}

			const temporary_proxy operator[]( const string_view& key ) const;
			const temporary_proxy operator[]( sint32 key ) const;
// TO FUCKING DO:
//			temporary_proxy operator[]( const string_view& key );
//			temporary_proxy operator[]( sint32 key );

		private:
			state* m_parent;
			int m_level;
			int m_index;
		};



	} // namespace lua

} // namespace nv

#endif // NV_LUA_STATE_HH
