// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/engine/model_manager.hh"

#include "nv/engine/ragdoll_manager.hh"
#include "nv/lua/lua_math.hh"
#include "nv/lua/lua_flags.hh"

using namespace nv;

bool nv::model_manager::load_resource( lua::table_guard& table, shash64 id )
{
	auto vec4_to_quat = [] ( const vec4& v ) { return quat( v.w, v.x, v.y, v.z ); };

	model* gm = new model;
	gm->flags  = table["flags"].as< flags<32> >();
	gm->attach = table["attach"].get_shash64();
	gm->root.set_position( table["root_position"].as<vec3>() );
	gm->root.set_orientation( vec4_to_quat( table["root_orientation"].as<vec4>( vec4(0.0f,0.0f,0.0f,1.0f) ) ) );
	
	resource< mesh_data > def_data;
	if ( table["path"].is_string() )
		def_data = m_rm->get< mesh_data >( table["path"].get_string128() );

	if ( table["ragdoll"].is_string() )
		gm->ragdoll_id = table["ragdoll"].get_string32();
	
	if ( table.has_field( "animator" ) )
	{
		gm->animator = m_rm->get< animator_data >( table["animator"].get_string() );
		pose_data_set* poses = gm->animator.lock()->poses;
		if ( !def_data || !def_data.lock()->node_data )
			gm->bind_data = m_animator_binds->add( id, new animator_bind_data( poses->get_tree(), poses->get_tree() ) );
		else
			gm->bind_data = m_animator_binds->add( id, new animator_bind_data( poses->get_tree(), *def_data.lock()->node_data ) );
	}

	if ( table.has_field( "phx_mesh" ) )
	{
		nv::string128 cpath( table["phx_mesh"].get_string128() );
		gm->phx_mesh = m_mesh_datas->get_path( cpath, def_data );
	}

	if ( table["model"].is_table() )
	{
		lua::table_guard model_table( table, "model" );
		read_model_node( model_table, gm, def_data );
	}

	add( id, gm );
	return true;
}



void nv::model_manager::read_model_node( lua::table_guard& table, model_node* node, resource< mesh_data > def_data )
{
	auto vec4_to_quat = [] ( const vec4& v ) { return quat( v.w, v.x, v.y, v.z ); };

	resource< data_channel_set > cmesh;
	resource< material >         cmaterial;
	sint16 attach_id = -1;

	if ( table["material"].is_string() )
	{
		nv::string128 mat_id( table["material"].get_string128() );
		cmaterial = m_rm->get< material >( mat_id );
		if ( !cmaterial )
			NV_LOG_ERROR( "Can't load material : ", mat_id );
	}

	if ( table["tag"].is_string() )
		node->tag = table["tag"].get_string32();

	if ( table.has_field( "path" ) )
	{
		nv::string128 cpath( table["path"].get_string128() );
		nv::data_node_info info;
		cmesh = m_mesh_datas->get_path( cpath, def_data, &info );
		attach_id = info.parent_id;
	}

	if ( table.has_field( "phx_hextents" ) )
		node->phx_hextents = table["phx_hextents"].as<vec3>();
	if ( table.has_field( "phx_offset" ) )
		node->phx_offset = table["phx_offset"].as<vec3>();
	if ( table.has_field( "phx_mass" ) )
		node->phx_mass = table["phx_mass"].as<float>( 0.0f );
	if ( table.has_field( "phx_mass" ) )
		node->phx_shape = nv::phx_shape( table["phx_shape"].as<int>( 0 ) );

	if ( table.has_field( "local_position" ) )
		node->local.set_position( table["local_position"].as<vec3>() );
	if ( table.has_field( "local_orientation" ) )
		node->local.set_orientation( vec4_to_quat( table["local_orientation"].as<vec4>( vec4( 0.0f, 0.0f, 0.0f, 1.0f ) ) ) );

	if ( table.has_field( "position" ) )
	{
		node->position.min = table["position"].as<vec3>();
		node->position.max = node->position.min;
	}
	if ( table.has_field( "rotation" ) )
	{
		node->rotation.min = table["rotation"].as<vec3>();
		node->rotation.max = node->rotation.min;
	}
	if ( table.has_field( "position_min" ) )  node->position.min  = table["position_min"].as<vec3>();
	if ( table.has_field( "position_max" ) )  node->position.max  = table["position_max"].as<vec3>();
	if ( table.has_field( "position_dist" ) ) node->position.dist = random_dist( table["position_dist"].get_uint32() );
	if ( table.has_field( "rotation_min" ) )  node->rotation.min  = table["rotation_min"].as<vec3>();
	if ( table.has_field( "rotation_max" ) )  node->rotation.max  = table["rotation_max"].as<vec3>();
	if ( table.has_field( "rotation_dist" ) ) node->rotation.dist = random_dist( table["rotation_dist"].get_uint32() );

	if ( table.has_field( "attach" ) )
	{
		if ( table["attach"].is_number() )
		{
			attach_id = sint16( table["attach"].get_sint32( -1 ) );
			//				parent_id = 0;
		}
		else if ( auto m = def_data.lock() )
		{
			auto it = m->node_names.find( table["attach"].get_shash64() );
			if ( it != m->node_names.end() )
				attach_id = sint16( it->second + 1 );
			int error; int hack;
		}
	}
	
	node->nflags    = table["flags"].as< flags<16,uint16> >();
	node->choice    = model_node_choice( table["choice"].get_uint32( 0 ) );
	node->chance    = table["chance"].get_f32( 1.0f );
	node->weight    = static_cast< uint16 >( table["weight"].get_uint32( 1 ) );
	node->mesh      = cmesh;
	node->attach_id = attach_id;
	node->material  = cmaterial;

	for ( uint32 i = 1; i <= table.get_size(); ++i )
	{
		lua::table_guard child_table( table, i );
		model_node* child = new model_node;
		node->children.push_back( child );
		read_model_node( child_table, child, def_data );
	}
}

