// Copyright (C) 2017-2017 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, 
// see copying.txt file in root folder.

/**
 * @file index_table.hh
 * @author Kornel Kisielewicz
 */

#ifndef NV_STL_INDEX_TABLE_HH
#define NV_STL_INDEX_TABLE_HH

#include <nv/common.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/hash_map.hh>

namespace nv
{

	template <
		typename Handle,
		typename Index = sint32
	>
		class index_table
	{
	public:
		typedef Handle              handle_type;
		typedef Index               index_type;

		index_table() {}
		index_table( uint32 reserve )
		{
			m_indexes.reserve( reserve );
		}

		index_type insert( handle_type h )
		{
			NV_ASSERT( !exists( h ), "Reinserting handle!" );
			resize_indexes_to( index_type( h.index() ) );
			index_type lindex = m_handles.size();
			m_indexes[h.index()] = index_type( lindex );
			m_handles.push_back( h );
			return lindex;
		}

		bool exists( handle_type h ) const
		{
			if ( h.is_nil() || h.index() >= m_indexes.size() ) return false;
			return m_indexes[h.index()] >= 0;
		}

		index_type get( handle_type h ) const
		{
			if ( h.is_nil() || h.index() >= m_indexes.size() ) return -1;
			return m_indexes[h.index()];
		}

		index_type remove_swap( handle_type h )
		{
			if ( h.is_nil() || h.index() >= m_indexes.size() || m_indexes[h.index()] == -1 )
				return -1;
			handle_type swap_handle = m_handles.back();
			index_type  dead_eindex = m_indexes[h.index()];
			if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
			{
				m_handles[unsigned( dead_eindex )] = swap_handle;
				m_indexes[swap_handle.index()] = dead_eindex;
			}
			m_handles.pop_back();
			m_indexes[h.index()] = -1;
			return dead_eindex;
		}

		index_type remove_swap( index_type dead_eindex )
		{
			if ( uint32( dead_eindex ) >= m_handles.size() ) return -1;
			handle_type h = m_handles[dead_eindex];
			handle_type swap_handle = m_handles.back();
			if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
			{
				m_handles[unsigned( dead_eindex )] = swap_handle;
				m_indexes[swap_handle.index()] = dead_eindex;
			}
			m_handles.pop_back();
			m_indexes[h.index()] = -1;
			return dead_eindex;
		}


		void clear()
		{
			m_handles.clear();
			m_indexes.clear();
		}

		handle_type get_handle( index_type i ) const { return m_handles[unsigned( i )]; }

		uint32 size() const { return m_handles.size(); }

	private:
		void resize_indexes_to( index_type i )
		{
			index_type size = index_type( m_indexes.size() );
			if ( i >= size )
			{
				if ( size == 0 ) size = 1;
				while ( i >= size ) size = size * 2;
				m_indexes.resize( static_cast<uint32>( size ), -1 );
			}
		}

		vector< handle_type > m_handles;
		vector< index_type >  m_indexes;
	};

	template <
		typename Handle,
		typename Index = sint32
	>
		class hashed_index_table
	{
	public:
		typedef Handle              handle_type;
		typedef Index               index_type;

		hashed_index_table() {}
		hashed_index_table( uint32 reserve )
		{
			m_indexes.reserve( reserve );
		}

		index_type insert( handle_type h )
		{
			NV_ASSERT( m_indexes.find( h ) == m_indexes.end(), "Reinserting handle!" );
			index_type lindex = m_handles.size();
			m_indexes[h] = index_type( lindex );
			m_handles.push_back( h );
			return lindex;
		}

		bool exists( handle_type h ) const
		{
			auto ih = m_indexes.find( h );
			return ( ih != m_indexes.end() );
		}

		index_type get( handle_type h ) const
		{
			auto ih = m_indexes.find( h );
			if ( ih == m_indexes.end() ) return -1;
			return ih->second;
		}

		index_type remove_swap( handle_type h )
		{
			if ( h.is_nil() ) return -1;
			auto ih = m_indexes.find( h );
			if ( ih == m_indexes.end() ) return -1;
			handle_type swap_handle = m_handles.back();
			index_type  dead_eindex = ih->second;
			if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
			{
				m_handles[unsigned( dead_eindex )] = swap_handle;
				m_indexes[swap_handle] = dead_eindex;
			}
			m_handles.pop_back();
			m_indexes.erase( h );
			return dead_eindex;
		}

// 		index_type remove_swap( index_type dead_eindex )
// 		{
// 			if ( uint32( dead_eindex ) >= m_handles.size() ) return -1;
// 			handle_type h = m_handles[dead_eindex];
// 			handle_type swap_handle = m_handles.back();
// 			if ( dead_eindex != static_cast<index_type>( m_handles.size() - 1 ) )
// 			{
// 				m_handles[unsigned( dead_eindex )] = swap_handle;
// 				m_indexes[swap_handle] = dead_eindex;
// 			}
// 			m_handles.pop_back();
// 			m_indexes.erase( h );
// 			return dead_eindex;
// 		}


		void clear()
		{
			m_handles.clear();
			m_indexes.clear();
		}

		handle_type get_handle( index_type i ) const { return m_handles[unsigned( i )]; }

		uint32 size() const { return m_handles.size(); }

	private:

		vector< handle_type > m_handles;
		hash_map< handle_type, index_type > m_indexes;
	};

}

#endif // NV_STL_INDEX_TABLE_HH
