// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file ecs.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief Data-driven Entity Component System
*/

#ifndef NV_ECS_HH
#define NV_ECS_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/index_table.hh>
#include <nv/stl/priority_queue.hh>
#include <nv/stl/handle_manager.hh>
#include <nv/core/types.hh>

namespace nv
{

	namespace ecs
	{


		template < typename Handle, typename Time = f32 >
		class ecs
		{
		public:
			typedef Time   time_type;
			typedef Handle handle_type;

			typedef uint32 message_type;

			struct message
			{
				message_type type;
				handle_type  entity;
				time_type    time;
				uint8        payload[128 - sizeof( uint32 ) - sizeof( handle_type ) - sizeof( time_type )];
			};

			struct message_compare_type
			{
				bool operator()( const message& l, const message& r )
				{
					return l.time > r.time;
				}
			};

			class receiver_interface
			{
			public:
				virtual void update( time_type dtime ) = 0;
				virtual bool handle_message( const message& ) = 0;
				virtual void clear() = 0;
			};

			class component_interface : public receiver_interface
			{
			public:
				virtual void remove( handle_type h ) = 0;
				virtual void* get_raw( handle_type h ) = 0;
				virtual const void* get_raw( handle_type h ) const = 0;
			};

			template < typename Payload >
			static const Payload& message_cast( const message& m )
			{
				static_assert( sizeof( Payload ) < sizeof( message::payload ), "Payload size over limit!" );
				NV_ASSERT( Payload::message_id == m.type, "Payload cast fail!" );
				return *reinterpret_cast<const Payload*>( &( m.payload ) );
			}

			void register_receiver( string_view /*name*/, receiver_interface* c )
			{
				m_receivers.push_back( c );
			}

			template < typename Component >
			void register_component( string_view name, component_interface* c )
			{
				m_components.push_back( c );
				register_receiver( name, c );
				m_component_map[thash64::create<Component>()] = c;
			}

			bool dispatch( const message& m )
			{
				for ( auto c : m_receivers )
					c->handle_message( m );
				return true;
			}

			template < typename Payload, typename ...Args >
			bool dispatch( handle_type h, Args&&... args )
			{
				message m{ Payload::message_id, h, time_type(0) };
				new( &m.payload ) Payload{ nv::forward<Args>( args )... };
				return dispatch( m );
			}

			bool queue( const message& m )
			{
				m_pqueue.push( m );
				return true;
			}

			template < typename Payload, typename ...Args >
			bool queue( handle_type h, time_type delay, Args&&... args )
			{
				message m{ Payload::message_id, h, m_time + delay };
				new( &m.payload ) Payload{ nv::forward<Args>( args )... };
				return queue( m );
			}

			handle_type create()
			{
				return m_handles.create_handle();
			}

			void update( time_type dtime )
			{
				if ( dtime == time_type(0) ) return;
				m_time += dtime;
				while ( !m_pqueue.empty() && m_pqueue.top().time <= m_time )
				{
					message msg = m_pqueue.top();
					m_pqueue.pop();
					dispatch( msg );
				}

				for ( auto c : m_receivers )
					c->update( dtime );
			}

			time_type update_step()
			{
				time_type before = m_time;
				if ( !m_pqueue.empty() )
				{
					message msg = m_pqueue.top();
					m_time = msg.time;
					m_pqueue.pop();
					dispatch( msg );
					if ( before != m_time )
						for ( auto c : m_receivers )
							c->update( m_time - before );
				}
				return m_time;
			}

			time_type get_time() const { return m_time; }

			bool events_pending() const
			{
				return !m_pqueue.empty();
			}

			const message& top_event() const
			{
				return m_pqueue.top();
			}

			void reset_events()
			{
				m_pqueue.clear();
				m_time = time_type(0);
			}

			void clear()
			{
				reset_events();
				for ( auto c : m_receivers )
					c->clear();
				m_handles.clear();
			}

			virtual void attach( handle_type parent, handle_type child )
			{
				m_handles.detach( child );
				m_handles.attach( parent, child );
			}

			handle_type get_parent( handle_type h ) const
			{
				return h ? m_handles.get_parent( h ) : handle_type();
			}

			handle_type next( handle_type h ) const
			{
				return h ? m_handles.next( h ) : handle_type();
			}

			handle_type first( handle_type h ) const
			{
				return h ? m_handles.first( h ) : handle_type();
			}

			void remove( handle_type h )
			{
				handle_type ch = m_handles.first( h );
				while ( handle_type r = ch )
				{
					ch = m_handles.next( ch );
					remove( r );
				}
				for ( auto c : m_components )
					c->remove( h );
				m_handles.free_handle( h );
			}

			bool exists( handle_type h ) const
			{
				return m_handles.is_valid( h );
			}

			template < typename Component >
			Component* get( handle_type h )
			{
				return static_cast< Component* >( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

			template < typename Component >
			const Component* get( handle_type h ) const
			{
				return static_cast<const Component*>( m_component_map[thash64::create< Component >()]->get_raw( h ) );
			}

		protected:

			handle_tree_manager< handle_type >          m_handles;
			vector< component_interface* >              m_components;
			vector< receiver_interface* >               m_receivers;
			hash_store< thash64, component_interface* > m_component_map;
			time_type                                   m_time = time_type(0);
			priority_queue< message, vector< message >, message_compare_type > m_pqueue;
		};

		template < typename Ecs, typename Component >
		class component : public Ecs::component_interface
		{
		public:
			typedef Ecs                                    ecs_type;
			typedef typename ecs_type::message             message_type;
			typedef typename ecs_type::handle_type         handle_type;
			typedef typename ecs_type::time_type           time_type;
			typedef index_table< handle_type >             index_table_type;
			typedef Component                              value_type;
			typedef Component                              component_type;
			typedef vector< value_type >                   storage_type;
			typedef typename index_table_type::index_type  index_type;

			typedef typename storage_type::iterator        iterator;
			typedef typename storage_type::const_iterator  const_iterator;
			typedef typename storage_type::reference       reference;
			typedef typename storage_type::const_reference const_reference;

			component( ecs_type& a_ecs, string_view a_name, uint32 reserve = 0 ) : m_ecs( a_ecs )
			{
				m_ecs.register_component<component_type>( a_name, this );
				if ( reserve != 0 )
				{
					m_data.reserve( reserve );
				}
			}

			value_type& insert( handle_type h )
			{
				index_type i = m_index.insert( h );
				NV_ASSERT( i == index_type( m_data.size() ), "Fail!" );
				NV_UNUSED( i );
				m_data.emplace_back();
				return m_data.back();
			}

			template < typename ...Args >
			value_type& insert( handle_type h, Args&&... args )
			{
				index_type i = m_index.insert( h );
				NV_ASSERT( i == index_type( m_data.size() ), "Fail!" );
				NV_UNUSED( i );
				m_data.emplace_back( nv::forward<Args>( args )... );
				return m_data.back();
			}

			bool exists( handle_type h )
			{
				return m_index.exists( h );
			}

			virtual void update( time_type /*dtime*/ )
			{
				// no-op
			}

			virtual void clear()
			{
				for ( uint32 i = 0; i < m_index.size(); ++i )
					destroy( &m_data[i] );
				m_index.clear();
				m_data.clear();
			}
			
			value_type* get( handle_type h )
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			const value_type* get( handle_type h ) const
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			void* get_raw( handle_type h )
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			const void* get_raw( handle_type h ) const
			{
				index_type i = m_index.get( h );
				return i >= 0 ? &( m_data[unsigned( i )] ) : nullptr;
			}

			virtual void remove( handle_type h )
			{
				value_type* v = get( h );
				if ( v == nullptr ) return;
				destroy( v );
				index_type dead_eindex = m_index.remove_swap( h );
				if ( dead_eindex == -1 ) return;
				if ( dead_eindex != static_cast<index_type>( m_data.size() - 1 ) )
				{
					m_data[unsigned( dead_eindex )] = move( m_data.back() );
				}
				m_data.pop_back();
			}

			virtual void destroy( value_type* )
			{
				// cleanup
			}

			virtual bool handle_message( const message_type& )
			{
				return true;
			}

			~component()
			{
				clear();
			}


			handle_type get_handle( index_type i ) const { return m_index.get_handle( i ); }

			const value_type& operator[] ( index_type i ) const { return m_data[i]; }
			value_type& operator[] ( index_type i ) { return m_data[i]; }

			size_t size() const { return m_data.size(); }

			iterator        begin() { return m_data.begin(); }
			const_iterator  begin()  const { return m_data.cbegin(); }
			const_iterator  cbegin() const { return m_data.cbegin(); }

			iterator        end() { return m_data.end(); }
			const_iterator  end()  const { return m_data.cend(); }
			const_iterator  cend() const { return m_data.cend(); }
		protected:
			ecs_type&        m_ecs;
			index_table_type m_index;
			storage_type     m_data;
		};

		template < typename Ecs >
		class receiver : public Ecs::receiver_interface
		{
		public:
			typedef Ecs                                    ecs_type;
			typedef typename ecs_type::message             message_type;
			typedef typename ecs_type::handle_type         handle_type;
			typedef typename ecs_type::time_type           time_type;

			receiver( ecs_type& a_ecs, string_view a_name ) : m_ecs( a_ecs )
			{
				m_ecs.register_receiver( a_name, this );
			}
			virtual void update( time_type /*dtime*/ )
			{
				// no-op
			}

			virtual void clear()
			{
				// no-op
			}
			virtual bool handle_message( const message_type& )
			{
				return false;
			}
		protected:
			ecs_type&        m_ecs;
		};
	}




}

#endif // NV_ECS_HH
