// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_ENGINE_PARTICLE_GROUP
#define NV_ENGINE_PARTICLE_GROUP

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/handle_store.hh>
#include <nv/interface/context.hh>

namespace nv
{

	enum class particle_orientation
	{
		POINT,
		ORIENTED,
		ORIENTED_COMMON,
		PERPENDICULAR,
		PERPENDICULAR_COMMON,
	};
	enum class particle_origin
	{
		CENTER,
		TOP_LEFT,
		TOP_CENTER,
		TOP_RIGHT,
		CENTER_LEFT,
		CENTER_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_CENTER,
		BOTTOM_RIGHT,
	};

	struct particle_vtx
	{
		vec3 position;
		vec2 texcoord;
		vec4 color;
	};

	struct particle_quad
	{
		particle_vtx data[6];
	};

	struct particle
	{
		vec3   position;
		vec4   color;
		vec3   velocity;
		vec2   size;
		vec2   tcoord_a;
		vec2   tcoord_b;
		float  rotation;
		float  lifetime;
		float  death;
	};

	struct particle_group_settings
	{
		vec3   common_up;
		vec3   common_dir;
		bool   accurate_facing;
		particle_orientation   orientation;
		particle_origin        origin;
	};

	struct particle_group_tag {};
	typedef handle< uint32, 16, 16, particle_group_tag > particle_group;

	struct particle_group_info
	{
		uint32                count;
		uint32                quota;
		vertex_array          vtx_array;
		buffer                vtx_buffer;
		bool                  local;
		particle_quad*        quads;
		uint32                ref_counter;
	};

	struct particle_render_data
	{
		uint32                count;
		vertex_array          vtx_array;
	};

	class particle_group_manager
	{
	public:
		particle_group_manager( context* a_context );
		particle_group create_group( uint32 max_particles );
		void release( particle_group group );
		void prepare( particle_group group );
		void reset();
		void generate_data(
			array_view< particle > particles,
			const particle_group_settings* data,
			particle_group group,
			const scene_state& s
		);
		particle_render_data get_render_data( particle_group group );
		~particle_group_manager();

		bool ref( particle_group group );
		bool unref( particle_group group );
	protected:
		void clear();

		context* m_context;
		handle_store< particle_group_info, particle_group > m_groups;
	};

}

NV_RTTI_DECLARE_NAME( nv::particle_orientation, "particle_orientation" )
NV_RTTI_DECLARE_NAME( nv::particle_origin, "particle_origin" )

#endif // NV_ENGINE_PARTICLE_GROUP