// Copyright (C) 2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file physics_world.hh
 * @author Kornel Kisielewicz
 * @brief physics_world
 */

#ifndef NV_PHYSICS_WORLD_HH
#define NV_PHYSICS_WORLD_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/core/transform.hh>
#include <nv/stl/memory.hh>

namespace nv
{
	class collision_shape
	{
	public:
		explicit collision_shape( void* p = nullptr, void* m = nullptr ) : internal( p ), mesh( m ) {}
		void* internal = nullptr;
		void* mesh     = nullptr;
	};


	class rigid_body
	{
	public:
		explicit rigid_body( void* p = nullptr ) : internal( p ) {}
		void* internal = nullptr;
	};

	class constraint
	{
	public:
		explicit constraint( void* p = nullptr ) : internal( p ) {}
		void* internal = nullptr;
	};

	enum collision_flags
	{
		PCF_STATIC_OBJECT = 1,
		PCF_KINEMATIC_OBJECT = 2,
		PCF_NO_CONTACT_RESPONSE = 4,
		PCF_CUSTOM_MATERIAL_CALLBACK = 8,
		PCF_CHARACTER_OBJECT = 16,
		PCF_DISABLE_VISUALIZE_OBJECT = 32,
		PCF_DISABLE_SPU_COLLISION_PROCESSING = 64
	};

	enum activation_state
	{
		PAS_ACTIVE_TAG = 1,
		PAS_ISLAND_SLEEPING = 2,
		PAS_WANTS_DEACTIVATION = 3,
		PAS_DISABLE_DEACTIVATION = 4,
		PAS_DISABLE_SIMULATION = 5,
	};

	class physics_world
	{
	public:
		virtual int step_simulation( float dtime, int max_sub_steps = 1, float fixed_time_step = 1.0f / 60.0f ) = 0;
		virtual collision_shape create_mesh( array_view< vec3 > vtx, array_view< uint32 > idx ) = 0;
		virtual collision_shape create_sphere( float radius ) = 0;
		virtual collision_shape create_capsule( float radius, float height ) = 0;
		virtual collision_shape create_cylinder( const vec3& half_extens ) = 0;
		virtual collision_shape create_box( const vec3& half_extens ) = 0;
		virtual collision_shape create_static_plane( const vec3& norm, float cst ) = 0;
		virtual void set_local_scaling( collision_shape shape, const vec3& value ) = 0;
		virtual vec3 get_local_scaling( collision_shape shape ) = 0;
		virtual rigid_body create_rigid_body( float mass, const transform& tr, collision_shape shape, const vec3& com_offset = vec3() ) = 0;
		virtual constraint create_hinge_constraint( rigid_body a, const transform& ta, rigid_body b, const transform& tb, const vec2& low_high, const vec3& params = vec3( 0.9f, 0.3f, 1.0f ) ) = 0;
		virtual constraint create_cone_twist_constraint( rigid_body a, const transform& ta, rigid_body b, const transform& tb, const vec3& sst, const vec3& params = vec3( 1.0f, 0.3f, 1.0f ) ) = 0;
		virtual void set_gravity( const vec3& ) = 0;
		virtual void add_rigid_body( rigid_body body ) = 0;
		virtual void remove_rigid_body( rigid_body body ) = 0;
		virtual void add_constraint( constraint cons ) = 0;
		virtual void set_rigid_body_damping( rigid_body body, float linear, float angular ) = 0;
		virtual void set_rigid_body_deactivation_time( rigid_body body, float time ) = 0;
		virtual void set_rigid_body_activation_state( rigid_body body, activation_state state ) = 0;
		virtual void set_rigid_body_sleeping_thresholds( rigid_body body, float linear, float angular ) = 0;
		virtual void set_rigid_body_linear_velocity( rigid_body body, const vec3& velocity ) = 0;
		virtual void set_rigid_body_ccd( rigid_body body, float radius, float threshold ) = 0;
		virtual vec3 get_rigid_body_linear_velocity( rigid_body body ) = 0;
		virtual transform get_world_transform( rigid_body body ) = 0;
		virtual void apply_central_impulse( rigid_body body, const vec3& impulse ) = 0;
		virtual void set_world_transform( rigid_body body, const transform& tr ) = 0;
		virtual int get_collision_flags( rigid_body body ) = 0;
		virtual void set_collision_flags( rigid_body body, int threshold ) = 0;
		virtual bool ray_test( const vec3& from, const vec3& to, vec3& hpos, vec3& hnorm, bool static_only = false ) = 0;
		virtual void release( collision_shape ) = 0;
		virtual void release( rigid_body ) = 0;
		virtual void release( constraint ) = 0;
	};

}

#endif // NV_PHYSICS_WORLD_HH
