// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/engine/model_manager.hh"

#include "nv/engine/ragdoll_manager.hh"
#include "nv/lua/lua_math.hh"
#include "nv/lua/lua_flags.hh"

using namespace nv;

bool nv::model_manager::load_resource( lua::table_guard& table, shash64 id )
{
	auto vec4_to_quat = [] ( const vec4& v ) { return quat( v.w, v.x, v.y, v.z ); };

	model* gm = new model;
	gm->flags  = table.get< flags<32> >( "flags" );
	gm->attach = table.get_string_hash_64( "attach" );
	gm->root.set_position( table.get<vec3>( "root_position", vec3() ) );
	gm->root.set_orientation( vec4_to_quat( table.get<vec4>( "root_orientation", vec4(0.0f,0.0f,0.0f,1.0f) ) ) );
	
	resource< mesh_data > def_data;
	if ( table.is_string( "path" ) )
		def_data = m_rm->get< mesh_data >( table.get_string128( "path" ) );

	if ( table.is_string( "ragdoll" ) )
		gm->ragdoll_id = table.get_string32( "ragdoll" );
	
	if ( table.has_field( "animator" ) )
	{
		gm->animator = m_rm->get< animator_data >( table.get_string( "animator" ) );
		pose_data_set* poses = gm->animator.lock()->poses;
		if ( !def_data || !def_data.lock()->node_data )
			gm->bind_data = m_animator_binds->add( id, new animator_bind_data( poses->get_tree(), poses->get_tree() ) );
		else
			gm->bind_data = m_animator_binds->add( id, new animator_bind_data( poses->get_tree(), *def_data.lock()->node_data ) );
	}

	if ( table.has_field( "phx_mesh" ) )
	{
		nv::string128 cpath( table.get_string128( "phx_mesh" ) );
		gm->phx_mesh = m_mesh_datas->get_path( cpath, def_data );
	}

	if ( table.is_table( "model" ) )
	{
		lua::table_guard model_table( table, "model" );
		read_model_node( model_table, gm, def_data );
	}

	add( id, gm );
	return true;
}



void nv::model_manager::read_model_node( lua::table_guard& table, model_node* node, resource< mesh_data > def_data )
{
	auto vec4_to_quat = [] ( const vec4& v ) { return quat( v.w, v.x, v.y, v.z ); };

	resource< data_channel_set > cmesh;
	resource< material >         cmaterial;
	sint16 attach_id = -1;

	if ( table.is_string( "material" ) )
	{
		nv::string128 mat_id( table.get_string128( "material" ) );
		cmaterial = m_rm->get< material >( mat_id );
		if ( !cmaterial )
			NV_LOG_ERROR( "Can't load material : ", mat_id );
	}

	if ( table.is_string( "tag" ) )
		node->tag = table.get_string32( "tag" );

	if ( table.has_field( "path" ) )
	{
		nv::string128 cpath( table.get_string128( "path" ) );
		nv::data_node_info info;
		cmesh = m_mesh_datas->get_path( cpath, def_data, &info );
		attach_id = info.parent_id;
	}

	if ( table.has_field( "local_position" ) )
		node->local.set_position( table.get<vec3>( "local_position", vec3() ) );
	if ( table.has_field( "local_orientation" ) )
		node->local.set_orientation( vec4_to_quat( table.get<vec4>( "local_orientation", vec4( 0.0f, 0.0f, 0.0f, 1.0f ) ) ) );

	if ( table.has_field( "position" ) )
	{
		node->position.min = table.get<vec3>( "position", vec3() );
		node->position.max = node->position.min;
	}
	if ( table.has_field( "rotation" ) )
	{
		node->rotation.min = table.get<vec3>( "rotation", vec3() );
		node->rotation.max = node->rotation.min;
	}
	if ( table.has_field( "position_min" ) )  node->position.min  = table.get<vec3>( "position_min", vec3() );
	if ( table.has_field( "position_max" ) )  node->position.max  = table.get<vec3>( "position_max", vec3() );
	if ( table.has_field( "position_dist" ) ) node->position.dist = random_dist( table.get_unsigned( "position_dist", 0 ) );
	if ( table.has_field( "rotation_min" ) )  node->rotation.min  = table.get<vec3>( "rotation_min", vec3() );
	if ( table.has_field( "rotation_max" ) )  node->rotation.max  = table.get<vec3>( "rotation_max", vec3() );
	if ( table.has_field( "rotation_dist" ) ) node->rotation.dist = random_dist( table.get_unsigned( "rotation_dist", 0 ) );

	if ( table.has_field( "attach" ) )
	{
		if ( table.is_number( "attach" ) )
		{
			attach_id = sint16( table.get_integer( "attach", -1 ) );
			//				parent_id = 0;
		}
		else if ( auto m = def_data.lock() )
		{
			auto it = m->node_names.find( table.get_string_hash_64( "attach" ) );
			if ( it != m->node_names.end() )
				attach_id = sint16( it->second + 1 );
			int error; int hack;
		}
	}
	
	node->nflags    = table.get< flags<16,uint16> >( "flags" );
	node->choice    = model_node_choice( table.get_unsigned( "choice", false ) );
	node->chance    = table.get_float( "chance", 1.0f );
	node->weight    = table.get_unsigned( "weight", 1 );
	node->mesh      = cmesh;
	node->attach_id = attach_id;
	node->material  = cmaterial;

	for ( uint32 i = 1; i <= table.get_size(); ++i )
	{
		lua::table_guard child_table( table, i );
		model_node* child = new model_node;
		node->children.push_back( child );
		read_model_node( child_table, child, def_data );
	}
}

