// Copyright (C) 2014-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_ENGINE_PARTICLE_ENGINE
#define NV_ENGINE_PARTICLE_ENGINE

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/unordered_map.hh>
#include <nv/stl/handle.hh>
#include <nv/stl/functional/function.hh>
#include <nv/lua/lua_state.hh>
#include <nv/gfx/texture_atlas.hh>
#include <nv/core/resource.hh>
#include <nv/core/random.hh>
#include <nv/interface/scene_node.hh>
#include <nv/interface/context.hh>
#include <nv/interface/physics_world.hh>
#include <nv/engine/particle_group.hh>

namespace nv
{

	static const unsigned MAX_PARTICLE_EMITTERS = 8;
	static const unsigned MAX_PARTICLE_AFFECTORS = 8;

	struct particle_emitter_data;
	struct particle_affector_data;

	typedef void( *particle_emitter_func )( random_base* r, const particle_emitter_data*, particle*, uint32 count );
	typedef void( *particle_affector_func )( const particle_affector_data*, particle*, float factor, uint32 count );
	typedef bool( *particle_affector_init_func )( lua::table_guard* table, particle_affector_data* data );

	struct particle_affector_funcs
	{
		particle_affector_func      process;
		particle_affector_init_func init;
	};

	struct particle_affector_data
	{
		particle_affector_func process;
		uint8 paramters[4 * 8 * 16];
	};

	struct particle_emitter_data
	{
		particle_emitter_func emitter_func;
		vec3   position;
		vec3   extents;
		vec3   hextents;
		vec3   iextents;
		vec3   ihextents;
		vec4   color_min;
		vec4   color_max;
		vec2   size_min;
		vec2   size_max;
		bool   precise;
		bool   square;
		float  angle;
		float  velocity_min;
		float  velocity_max;
		float  lifetime_min;
		float  lifetime_max;
		vec3   dir;
		vec3   odir;
		vec3   cdir;
		float  duration_min;
		float  duration_max;
		float  repeat_min;
		float  repeat_max;
		float  rate;
	};

	struct particle_emitter_info
	{
		bool   active;
		float  pause;
		float  next;
	};

	struct particle_system_data : particle_group_settings
	{
		uint32                       quota;
		uint32                       emitter_count;
		particle_emitter_data        emitters[MAX_PARTICLE_EMITTERS];
		uint32                       affector_count;
		particle_affector_data       affectors[MAX_PARTICLE_AFFECTORS];
	};

	struct particle_system_tag {};
	typedef handle< uint32, 16, 16, particle_system_tag > particle_system;
//	using particle_on_collide_func = function<bool( vec3, vec3 )>;
	using particle_on_collide_func = function<void( const vec3&, const vec3& )>;

	struct particle_system_info
	{
		particle*                particles;
		particle_emitter_info    emitters[MAX_PARTICLE_EMITTERS];
		particle_on_collide_func on_collide;

		uint32         count;
		vec2           texcoords[2];
		particle_group group;


		const particle_system_data*    data;
	};

}

NV_RTTI_DECLARE_NAME( nv::particle_system_data, "particle_system_data" )

namespace nv
{

	class particle_engine
	{
	public:
		particle_engine( context* a_context, random_base* rng, particle_group_manager* groups, bool debug_data = false );
		void reset();
		void prepare( particle_group group );
		particle_group create_group( uint32 max_particles );
		particle_system create_system( resource< particle_system_data > rdata, particle_group group );
		particle_system create_system( const particle_system_data* data, particle_group group );
		void release( particle_group group );
		void release( particle_system system );
		bool is_finished( particle_system system );
		void update( particle_system system, transform model, float dtime );
		void render( particle_system system, const scene_state& s );
		void set_on_collide( particle_system system, const particle_on_collide_func& f );
		void set_texcoords( particle_system system, vec2 a, vec2 b );
		static void register_emitter_type( const string_view& name, particle_emitter_func func );
		static void register_affector_type( const string_view& name, particle_affector_init_func init, particle_affector_func process );
		static particle_emitter_func   get_emitter( shash64 emitter );
		static particle_affector_funcs get_affector( shash64 affector );
		static string_view get_debug_name( particle_emitter_func f );
		static string_view get_debug_name( particle_affector_func f );
		static const vector< particle_emitter_func >* get_debug_emitter_list() { return m_debug_emitter_list; }
		static const vector< particle_affector_func >* get_debug_affector_list() { return m_debug_affector_list; }
		static const type_entry* get_debug_type( particle_affector_func f );
		static void register_types( type_database* db, bool debug_data = false );
		particle_render_data get_render_data( particle_group group );
		particle_group_manager* get_particle_manager() { return m_pgm; }
		~particle_engine();
	private:
		void clear();
		void register_standard_emitters();
		void register_standard_affectors();
		void destroy_particles( particle_system_info* info, float dtime );
		void create_particles( particle_system_info* info, transform tr, float dtime );
		void update_particles( particle_system_info* info, float dtime );
		void update_emitters( particle_system_info* info, float dtime );

		particle_group_manager* m_pgm;
		context* m_context;
		random_base* m_rng;

		handle_store< particle_system_info, particle_system >             m_systems;

		static hash_store< shash64, particle_emitter_func >          m_emitters;
		static hash_store< shash64, particle_affector_funcs >        m_affectors;

		static vector< particle_emitter_func >*  m_debug_emitter_list;
		static vector< particle_affector_func >* m_debug_affector_list;
		static hash_map< particle_emitter_func,  const_string >* m_debug_emitter_names;
		static hash_map< particle_affector_func, const_string >* m_debug_affector_names;
		static hash_map< particle_affector_func, type_entry* >*  m_debug_affector_types;

	};

}

#endif // NV_ENGINE_PARTICLE_ENGINE
