// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file animation.hh
 * @author Kornel Kisielewicz
 * @brief animation
 */

#ifndef NV_ENGINE_ANIMATION_HH
#define NV_ENGINE_ANIMATION_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/interface/easing.hh>
#include <nv/gfx/skeleton_instance.hh>
#include <nv/gfx/poses.hh>
#include <nv/engine/resource_system.hh>

namespace nv
{
	class animator_bind_data;
	struct animator_data;
}

NV_RTTI_DECLARE_NAME ( nv::mesh_nodes_data,    "nv::mesh_nodes_data"    )
NV_RTTI_DECLARE_NAME ( nv::animator_bind_data, "nv::animator_bind_data" )
NV_RTTI_DECLARE_NAME ( nv::animator_data,      "nv::animator_data"      )

namespace nv
{

	struct animator_pose_data
	{
		uint32 pose;
		// float  time;
		//	float      easing_data[4];
	};

	struct animator_transition_data
	{
		uint32        target;
		float         duration;
		interpolation interp;
		easing        easing;
	};

	typedef hash_store< shash64, animator_transition_data > animator_transition_table;

	struct animator_state_data
	{
		shash64                      name;
		uint32                       base_pose;
		vector< animator_pose_data > poses;
		bool                         loop;
		float                        duration;
		interpolation                interp;
		animator_transition_table    transitions;
	};

	struct animator_layer_data
	{
		shash64 name;
		sint32  def_state;
		sint32  mask;
		vector< bool >                mask_vector;
		vector< animator_state_data > states;
	};

	struct animator_data
	{
		vector< animator_layer_data > layers;
		pose_data_set*                poses;
		string64                      id;
	};

	struct animator_transition_instance : animator_transition_data
	{
		uint32        source;
		float         time;

		animator_transition_instance() {}

		animator_transition_instance( const animator_transition_data& data, uint32 src )
			: animator_transition_data( data ), source( src ), time( 0.0f )
		{

		}
	};

	class animator_layer_instance
	{
	public:
		vector< animator_transition_instance > m_transitions;
		sint32 m_current_state;
		float m_time;

		animator_layer_instance() : m_current_state( 0 ), m_time( 0.0f ) {}
		void set_state( uint32 state ) { m_current_state = sint32( state ); }
		sint32 get_state() const { return m_current_state; }
		sint32 get_final_state() const
		{
			if ( m_transitions.size() > 0 ) 
				return sint32( m_transitions.back().target );
			return m_current_state;
		}
		float add_transition( const animator_transition_data& data, uint32 source )
		{
			float duration = 0.0f;
			if ( m_transitions.size() > 0 )
			{
				for ( const auto& t : m_transitions )
					duration += t.duration - t.time;
				auto& last = m_transitions.back();
				if ( last.target == source && data.target == last.source )
				{
					nv::swap( last.target, last.source );
					last.time = last.duration - last.time;
					/* this duration might be inaccurate? */
					return duration;
				}
			}
			m_transitions.push_back( animator_transition_instance( data, source ) );
			return duration + m_transitions.back().duration;
		}
		void reset()
		{
			m_transitions.clear();
			m_time = 0.0f;
		}
		void update( float dtime )
		{
			if ( m_current_state != -1 )
			{
				float time = dtime;
				while ( m_transitions.size() > 0 )
				{
					animator_transition_instance& tr = m_transitions.front();
					m_current_state = sint32( tr.target );
					tr.time += time;
					if ( tr.time < tr.duration ) break;
					time -= ( tr.time - tr.duration );
					m_transitions.erase( m_transitions.begin() );
				}
				m_time += dtime;
			}
		}
	};

	// resource per animator/model set
	class animator_bind_data
	{
	public:
		animator_bind_data( const data_node_list& pose_frame, const data_node_list& bones )
			: m_bone_list( bones.get_name() )
		{
			m_bone_list.assign( bones );
			m_bone_transforms.prepare( bones );
			m_pose_binding.prepare( pose_frame, bones );
		}
		const bone_transforms< transform >& get_bone_transforms() const { return m_bone_transforms; }
		const skeleton_binding& get_pose_binding() const
		{
			return m_pose_binding;
		}
		const data_node_list& get_bone_list() const { return m_bone_list;  }

	protected:
		skeleton_binding                        m_pose_binding;
		bone_transforms< transform >            m_bone_transforms;
		data_node_list                          m_bone_list;
	};

	// per model instance
	class animator_instance
	{
	public:
		animator_instance() {}
		void initialize( const animator_data* data )
		{
			m_layers.clear();
			m_layers.resize( data->layers.size() );
			for ( uint32 i = 0; i < data->layers.size(); ++i )
			{
				m_layers[i].set_state( uint32( data->layers[i].def_state ) );
			}
		}

		const skeleton_transforms& get_transforms() const
		{
			return m_transforms;
		}

		sint32 base_state() const
		{
			// HACK
			if ( m_layers.size() > 1 && m_layers[0].m_transitions.size() < 1 ) return m_layers[0].get_state();
			return 0;
		}
		
		float queue_event( const animator_data* data, shash64 name )
		{
			float duration = 0.0f;
			for ( uint32 i = 0; i < m_layers.size(); ++i )
				// what about activating nonactive layers?
				if ( m_layers[i].m_current_state != -1 )
				{
					sint32 last_state = m_layers[i].get_final_state();
					const animator_state_data& state = data->layers[i].states[ uint32( last_state ) ];
					auto it = state.transitions.find( name );
					if ( it != state.transitions.end() )
					{
						float d = m_layers[i].add_transition( it->second, uint32( last_state ) );
						duration = nv::max( d, duration );
					}
				}
			return duration;
		}

		void update( const animator_data* data, float dtime, bool do_delocalize = true )
		{
			for ( uint32 i = 0; i < m_layers.size(); ++i )
				m_layers[i].update( dtime );

			for ( uint32 i = 0; i < m_layers.size(); ++ i )
				if ( m_layers[i].m_current_state != -1 )
				{
					if ( m_layers[i].m_transitions.size() > 0 )
					{
						animate_layer( data->layers[i], m_layers[i].m_transitions.front(), data->poses, m_layers[i].m_time );
					}
					else
					{
						const animator_layer_data& layer = data->layers[i];
						const animator_state_data& state = layer.states[ uint32( m_layers[i].m_current_state ) ];
						if ( state.poses.size() > 0 )
							animate_state( m_transforms, state, data->layers[i], data->poses, get_frame( state, m_layers[i].m_time ) );
					}
				}

			if ( do_delocalize )
				m_transforms.delocalize( data->poses->get_tree() );
		}
		void delocalize( const animator_data* data )
		{
			m_transforms.delocalize( data->poses->get_tree() );
		}

		void reset()
		{
			for ( uint32 i = 0; i < m_layers.size(); ++i )
				m_layers[i].reset();
		}

		void update_state_only( const animator_data* data, float dtime, uint32 layer_id, uint32 state_id )
		{
			const animator_layer_data& layer = data->layers[layer_id];
			const animator_state_data& state = layer.states[state_id];
			m_layers[layer_id].update( dtime );
			if ( state.poses.size() > 0 )
				animate_state( m_transforms, state, layer, data->poses, get_frame( state, m_layers[layer_id].m_time ) );
			m_transforms.delocalize( data->poses->get_tree() );
		}

		void update_layer_only( const animator_data* data, float dtime, uint32 layer_id )
		{
			const animator_layer_data& ldata     = data->layers[layer_id];
			const animator_layer_instance& linst = m_layers[layer_id];
			m_layers[layer_id].update( dtime );
			if ( linst.m_current_state != -1 )
			{
				if ( m_layers[layer_id].m_transitions.size() > 0 )
				{
					animate_layer( ldata, linst.m_transitions.front(), data->poses, linst.m_time );
				}
				else
				{
					const animator_state_data& state = ldata.states[ uint32( linst.m_current_state ) ];
					if ( state.poses.size() > 0 )
						animate_state( m_transforms, state, ldata, data->poses, get_frame( state, linst.m_time ) );
				}
			}
			m_transforms.delocalize( data->poses->get_tree() );
		}

		const skeleton_transforms& stored()
		{
			return m_stored;
		}

		void store()
		{
			m_stored.assign( m_transforms );
		}


//	protected:

		void animate_layer( const animator_layer_data& layer, const animator_transition_instance& transition, pose_data_set* pose_data, float t )
		{
			const animator_state_data& base_state = layer.states[transition.source];
			if ( base_state.poses.size() > 0 )
				animate_state( m_transforms, base_state, layer, pose_data, get_frame( base_state, t ) );

			float trblend = 0.0f;
			if ( transition.duration > 0.0f )
			{
				trblend = math::clamp( transition.time / transition.duration, 0.0f, 1.0f );
				trblend = easing_eval( trblend, transition.easing );
			}

			const animator_state_data& state = layer.states[transition.target];
			if ( state.poses.size() > 0 )
				animate_state( m_transforms, state, layer, pose_data, get_frame( state, t ), true, trblend, transition.interp );
		}

		static float get_frame( const animator_state_data& state, float layer_time )
		{
			uint32 fcount = state.poses.size();
			float fframe = 0.0f;
			if ( state.duration > nv::math::epsilon<float>() )
			{
				float fps = fcount / state.duration;
				float ctime = state.loop ? fmodf( layer_time, state.duration ) : min( layer_time, state.duration - 0.0001f );
				fframe = ctime * fps; 
			}
			return fframe;
		}

		static void animate_state( skeleton_transforms& transforms, const animator_state_data& state, const animator_layer_data& layer, const pose_data_set* pose_data, float fframe, bool do_blend = false, float blend = 0.0f, nv::interpolation binterp = interpolation::NONE )
		{
			unsigned v1 = unsigned( fframe );
			unsigned v2 = v1 + 1;
			if ( v2 >= state.poses.size() ) v2 = state.loop ? 0 : state.poses.size() - 1;

			unsigned iv1 = state.poses[v1].pose;
			unsigned iv2 = state.poses[v2].pose;


			if ( iv1 == iv2 )
			{
				if ( do_blend )
					transforms.interpolate( transforms, pose_data->get( iv1 ), blend, binterp, layer.mask_vector );
				else
					transforms.assign( pose_data->get( iv1 ), layer.mask_vector );
			}
			else
			{
				if ( state.interp == interpolation::QUADRATIC || state.interp == interpolation::SQUADRATIC )
				{
					unsigned s1 = v1 == 0 ? state.poses.size() - 1 : v1 - 1; 
					unsigned s2 = v2 + 1;
					if ( s2 >= state.poses.size() ) s2 = state.loop ? 0 : state.poses.size() - 1;
					unsigned is1 = state.poses[s1].pose;
					unsigned is2 = state.poses[s2].pose;
					if ( do_blend )
						transforms.blend( pose_data->get( is1 ), pose_data->get( iv1 ), pose_data->get( iv2 ), pose_data->get( is2 ), fframe - v1, state.interp, blend, binterp, layer.mask_vector );
					else
						transforms.interpolate( pose_data->get( is1 ), pose_data->get( iv1 ), pose_data->get( iv2 ), pose_data->get( is2 ), fframe - v1, state.interp, layer.mask_vector );
				}
				else
				{
					if ( do_blend )
						transforms.blend( pose_data->get( iv1 ), pose_data->get( iv2 ), fframe - v1, state.interp, blend, binterp, layer.mask_vector );
					else
						transforms.interpolate( pose_data->get( iv1 ), pose_data->get( iv2 ), fframe - v1, state.interp, layer.mask_vector );
				}	
			}
		}


		vector< animator_layer_instance > m_layers;
		skeleton_transforms               m_transforms;
		skeleton_transforms               m_stored;
	};

	class animator_manager : public file_resource_manager< animator_data, true, lua_resource_manager_base >
	{
	public:
		explicit animator_manager( nv::string_table* strings = nullptr ) : m_strings( strings ) {}
		virtual nv::string_view get_storage_name() const { return "animators"; }
		virtual nv::string_view get_resource_name() const { return "animator"; }

		nv::resource< animator_data > load_animator( const nv::string_view& id, nv::pose_data_set* poses = nullptr );
		nv::resource< animator_data > create_animator( const nv::string_view& id, nv::pose_data_set* poses = nullptr );
	protected:
		virtual bool load_resource( nv::lua::table_guard& table, nv::shash64 id );
		nv::resource< animator_data > load_animator( nv::lua::table_guard& table, nv::shash64 id, nv::pose_data_set* poses = nullptr );

		nv::easing read_easing( nv::lua::table_guard& table );
		void fill_mask_vector_rec( nv::vector< bool >& mask_vector, const nv::data_node_tree& tree, nv::uint32 id )
		{
			mask_vector[id] = true;
			for ( auto child : tree.children( id ) )
				fill_mask_vector_rec( mask_vector, tree, child );
		}
		
		nv::string_table* m_strings;
	};

	class animator_bind_manager : public nv::manual_resource_manager< nv::animator_bind_data >
	{
	public:
		animator_bind_manager() {}
		using nv::manual_resource_manager< nv::animator_bind_data >::add;
		using nv::manual_resource_manager< nv::animator_bind_data >::exists;
	protected:
	};

}

#endif // NV_ENGINE_ANIMATION_HH
