// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_ENGINE_PARTICLE_ENGINE
#define NV_ENGINE_PARTICLE_ENGINE

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/unordered_map.hh>
#include <nv/stl/handle.hh>
#include <nv/lua/lua_state.hh>
#include <nv/gfx/texture_atlas.hh>
#include <nv/core/resource.hh>
#include <nv/interface/scene_node.hh>
#include <nv/interface/context.hh>

namespace nv
{

	static const unsigned MAX_PARTICLE_EMITTERS = 8;
	static const unsigned MAX_PARTICLE_AFFECTORS = 8;

	struct particle_emitter_data;
	struct particle_affector_data;
	struct particle;

	typedef void( *particle_emitter_func )( const particle_emitter_data*, particle*, uint32 count );
	typedef void( *particle_affector_func )( const particle_affector_data*, particle*, float factor, uint32 count );
	typedef bool( *particle_affector_init_func )( lua::table_guard* table, particle_affector_data* data );

	struct particle_affector_funcs
	{
		particle_affector_func      process;
		particle_affector_init_func init;
	};

	enum class particle_orientation
	{
		POINT,
		ORIENTED,
		ORIENTED_COMMON,
		PERPENDICULAR,
		PERPENDICULAR_COMMON,
	};
	enum class particle_origin
	{
		CENTER,
		TOP_LEFT,
		TOP_CENTER,
		TOP_RIGHT,
		CENTER_LEFT,
		CENTER_RIGHT,
		BOTTOM_LEFT,
		BOTTOM_CENTER,
		BOTTOM_RIGHT,
	};

	struct particle_vtx
	{
		vec3 position;
		vec2 texcoord;
		vec4 color;
	};

	struct particle_quad
	{
		particle_vtx data[6];
	};

	struct particle
	{
		vec3   position;
		vec4   color;
		vec3   velocity;
		vec2   size;
		vec2   tcoord_a;
		vec2   tcoord_b;
		float  rotation;
		float  lifetime;
		float  death;
	};

	struct particle_affector_data
	{
		particle_affector_func process;
		uint8 paramters[4 * 8 * 16];
	};

	struct particle_emitter_data
	{
		particle_emitter_func emitter_func;
		vec3   position;
		vec3   extents;
		vec3   hextents;
		vec3   iextents;
		vec3   ihextents;
		vec4   color_min;
		vec4   color_max;
		vec2   size_min;
		vec2   size_max;
		bool   precise;
		bool   square;
		float  angle;
		float  velocity_min;
		float  velocity_max;
		float  lifetime_min;
		float  lifetime_max;
		vec3   dir;
		vec3   odir;
		vec3   cdir;
		float  duration_min;
		float  duration_max;
		float  repeat_min;
		float  repeat_max;
		float  rate;
	};

	struct particle_emitter_info
	{
		bool   active;
		float  pause;
		float  next;
	};

	struct particle_system_data
	{
		uint32 quota;
		vec3   common_up;
		vec3   common_dir;
		bool   accurate_facing;
		particle_orientation   orientation;
		particle_origin        origin;
		uint32                 emitter_count;
		particle_emitter_data  emitters[MAX_PARTICLE_EMITTERS];
		uint32                 affector_count;
		particle_affector_data affectors[MAX_PARTICLE_AFFECTORS];
	};

	struct particle_system_group_tag {};
	typedef handle< uint32, 16, 16, particle_system_group_tag > particle_system_group;
	struct particle_system_tag {};
	typedef handle< uint32, 16, 16, particle_system_tag > particle_system;

	struct particle_system_info
	{
		particle*             particles;
		particle_emitter_info emitters[MAX_PARTICLE_EMITTERS];

		uint32                count;
		vec2                  texcoords[2];
		particle_system_group group;

		const particle_system_data*    data;
	};

	struct particle_system_group_info
	{
		uint32                count;
		uint32                quota;
		vertex_array          vtx_array;
		buffer                vtx_buffer;
		bool                  local;
		particle_quad*        quads;
		uint32                ref_counter;
	};

	struct particle_render_data
	{
		uint32                count;
		vertex_array          vtx_array;
	};

}

NV_RTTI_DECLARE_NAME( nv::particle_system_data, "particle_system_data" )
NV_RTTI_DECLARE_NAME( nv::particle_orientation, "particle_orientation" )
NV_RTTI_DECLARE_NAME( nv::particle_origin,      "particle_origin" )

namespace nv {

	class particle_engine
	{
	public:
		particle_engine( context* a_context, bool debug_data = false );
		void reset();
		void prepare( particle_system_group group );
		particle_system_group create_group( uint32 max_particles );
		particle_system create_system( resource< particle_system_data > rdata, particle_system_group group );
		particle_system create_system( const particle_system_data* data, particle_system_group group );
		void release( particle_system_group group );
		void release( particle_system system );
		void update( particle_system system, transform model, float dtime );
		void render( particle_system system, const scene_state& s );
		void set_texcoords( particle_system system, vec2 a, vec2 b );
		static void register_emitter_type( const string_view& name, particle_emitter_func func );
		static void register_affector_type( const string_view& name, particle_affector_init_func init, particle_affector_func process );
		static particle_emitter_func   get_emitter( shash64 emitter );
		static particle_affector_funcs get_affector( shash64 affector );
		static string_view get_debug_name( particle_emitter_func f );
		static string_view get_debug_name( particle_affector_func f );
		static const type_entry* get_debug_type( particle_affector_init_func f );
		static void register_types( type_database* db );
		particle_render_data get_render_data( particle_system_group group );
		~particle_engine();
	private:
		void clear();
		void register_standard_emitters();
		void register_standard_affectors();
		void generate_data( particle_system_info* info, const scene_state& s );
		void destroy_particles( particle_system_info* info, float dtime );
		void create_particles( particle_system_info* info, transform tr, float dtime );
		void update_particles( particle_system_info* info, float dtime );
		void update_emitters( particle_system_info* info, float dtime );

		context* m_context;

		handle_store< particle_system_info, particle_system >             m_systems;
		handle_store< particle_system_group_info, particle_system_group > m_groups;

		static hash_store< shash64, particle_emitter_func >          m_emitters;
		static hash_store< shash64, particle_affector_funcs >        m_affectors;

		static hash_map< particle_emitter_func,  const_string >* m_debug_emitter_names;
		static hash_map< particle_affector_func, const_string >* m_debug_affector_names;
		static hash_map< particle_affector_func, type_entry* >*  m_debug_affector_types;

	};

}

#endif // NV_ENGINE_PARTICLE_ENGINE
