// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file terminal.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief terminal interface
*/

#ifndef NV_CORE_TERM_COLOR_HH
#define NV_CORE_TERM_COLOR_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>

namespace nv
{

	class term_color
	{
	public:
		enum color
		{
			BLACK,
			BLUE,
			GREEN,
			CYAN,
			RED,
			MAGENTA,
			BROWN,
			LIGHTGRAY,
			DARKGRAY,
			LIGHTBLUE,
			LIGHTGREEN,
			LIGHTCYAN,
			LIGHTRED,
			LIGHTMAGENTA,
			YELLOW,
			WHITE,
			TRANSPARENT,
		};
		term_color() : term_color( BLACK ) {}
		/*implicit */term_color( color c ) : term_color( s_default_colors[c] ) {}
		term_color( color c, u8vec3 rgb ) : rgb( rgb ), c( uint8( c ) ), a( 15 ) {}
		term_color( color c, u8vec3 rgb, uint32 a ) : rgb( rgb ), c( uint8( c ) ), a( uint8( a ) ) {}
		term_color( color c, u8vec4 rgba ) : rgb( rgba.r, rgba.g, rgba.b ), c( uint8( c ) ), a( rgba.a ) {}

		u8vec3 get_rgb() const { return rgb; }
		u8vec4 get_rgba() const { return u8vec4( rgb, a * 17 ); }
		vec3 get_frgb() const { return vec3( rgb.r / 255.0f, rgb.g / 255.0f, rgb.b / 255.0f ); }
		vec4 get_frgba() const { return vec4( get_rgb(), a / 15.0f ); }
		uint32 get_color() const { return c; }

		uint32 get_argb32() const { return ( ( a * 17 ) << 24 ) + ( rgb.r << 16 ) + ( rgb.g << 8 ) + rgb.b; }
		uint32 get_rgb32() const { return ( rgb.r << 16 ) + ( rgb.g << 8 ) + rgb.b; }

	protected:
		union
		{
			u8vec4 rgb_ca;
			struct
			{
				u8vec3 rgb;
				uint8  c : 4;
				uint8  a : 4;
			};
		};
		static term_color s_default_colors[17];
	};

	static_assert( sizeof( term_color ) == 4, "Packing issues!" );

}

#endif // NV_CORE_TERM_COLOR_HH
