// Copyright (C) 2016-2016 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file model_manager.hh
* @author Kornel Kisielewicz
* @brief model_manager
*/

#ifndef NV_ENGINE_MODEL_MANAGER_HH
#define NV_ENGINE_MODEL_MANAGER_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/core/random.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/engine/material_manager.hh>
#include <nv/engine/mesh_manager.hh>
#include <nv/engine/animation.hh>
#include <nv/engine/resource_system.hh>

#define MNODE 1

namespace nv
{

	struct model_node
	{
		resource< data_channel_set > mesh;
		resource< material >         material;
		transform                    local;
		sint16                       attach_id;
		float                        chance;
		bool                         random_rotate_y;
		bool                         force;
		vector< model_node* >        children;

		model_node() : attach_id( 0 ), chance( 1.0f ), random_rotate_y( false ), force( false ) {}

		~model_node()
		{
			for ( auto c : children ) delete c;
		}
	};

	struct model : model_node
	{
		resource< animator_data >      animator;
		resource< animator_bind_data > bind_data;
		transform                      root;
		shash64                        attach;
 	};

	struct flat_model_element
	{
		resource< data_channel_set > mesh;
		resource< material >         material;
		sint16                       attach_id;
		sint16                       parent_id;
		transform					 local;
	};

	struct flat_model
	{
		flat_model_element             elements[8];
		uint32                         count;
		shash64                        attach;
		transform					   local;
	};

	NV_RTTI_DECLARE_NAME( model, "model" )

	class model_manager : public lua_resource_manager< model >
	{
	public:
		model_manager( resource_manager* rm, animator_bind_manager* binds, mesh_data_manager* mesh_datas )
			: m_rm( rm )
			, m_animator_binds( binds )
			, m_mesh_datas( mesh_datas )
		{
		}
		virtual string_view get_storage_name() const { return "models"; }
		virtual string_view get_resource_name() const { return "model"; }
		static flat_model flatten( const model* m, random_base& rng )
		{
			flat_model result;
			if ( !m ) return result;
			result.attach = m->attach;
			result.local = m->root;
			result.count = 0;
			flatten( result, m, rng, transform(), -1 );
			return result;
		}
	protected:
		static void flatten( flat_model& result, const model_node* m, random_base& rng, const transform& ptr, sint32 parent_id )
		{
			if ( m->chance < 1.0f )
				if ( rng.frand() > m->chance )
					return;
			transform tr = ptr * m->local;
			if ( m->random_rotate_y )
				tr = tr * transform( math::angle_axis( nv::math::pi<float>() * 2.0f * rng.frand(), vec3( 0.0f, 1.0f, 0.0f ) ) );

			if ( m->mesh || m->force )
			{
				uint32 id = result.count++;
				flat_model_element& re = result.elements[id];
				re.mesh = m->mesh;
				re.material = m->material;
				re.local = tr;
				re.parent_id = parent_id;
				re.attach_id = m->attach_id;
				parent_id = id;
			}
			for ( auto c : m->children )
			{
				flatten( result, c, rng, tr, parent_id );
			}
		}

	protected:
		virtual bool load_resource( lua::table_guard& table, shash64 id );
		void read_model_node( lua::table_guard& table, model_node* node, resource< mesh_data > def_data );
	private:
		resource_manager*      m_rm;
		animator_bind_manager* m_animator_binds;
		mesh_data_manager*     m_mesh_datas;
	};

}

#endif // NV_ENGINE_MODEL_MANAGER_HH
