// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file resource_manager.hh
 * @author Kornel Kisielewicz
 * @brief engine resource manager
 */

#ifndef NV_ENGINE_RESOURCE_MANAGER_HH
#define NV_ENGINE_RESOURCE_MANAGER_HH

#include <nv/common.hh>
#include <nv/interface/context.hh>
#include <nv/core/resource.hh>
#include <nv/lua/lua_state.hh>
#include <nv/stl/hash_store.hh>
#include <nv/stl/vector.hh>

namespace nv
{

	template < typename T, bool Heap = true >
	struct resource_storage_policy;

	template < typename T >
	struct resource_storage_policy< T, true >
	{
		typedef T* type;
		static void free( T* value ) { delete value; }
		static T* to_pointer( T* value ) { return value; }
		static T* to_stored( void* value ) { return reinterpret_cast<T*>( value ); }
	};

	template < typename T >
	struct resource_storage_policy< T, false >
	{
		typedef T type;
		static void free( T ) {}
		static T* to_pointer( T& value ) { return &value; }
		static T to_stored( void* value ) { return *reinterpret_cast<T*>( value ); }
	};

	class resource_system;

	class lua_resource_manager_base : public resource_handler
	{
	public:
		lua_resource_manager_base( resource_type_id id ) : resource_handler( id ), m_lua( nullptr ) {}
		void initialize( lua::state* state );
		virtual string_view get_storage_name() const = 0;
		virtual string_view get_resource_name() const = 0;
		virtual void clear() = 0;
		void load_all( bool do_clear = true );
		virtual bool load_resource( const string_view& id );
		virtual ~lua_resource_manager_base() {}
	protected:
		virtual bool load_resource( lua::table_guard& table, shash64 id ) = 0;
		lua::state* m_lua;
	};

	template < typename T, bool Heap = true, typename Base = resource_handler >
	class custom_resource_manager : public Base
	{
	public:
		typedef resource_storage_policy< T, Heap > policy_type;
		typedef Base                       base_type;
		typedef T                          value_type;
		typedef resource< T >              resource_type;
		typedef typename policy_type::type stored_type;

		custom_resource_manager() : base_type( resource_type_id( rtti_type_hash<T>::hash() ) ) {}
		resource_type get( const string_view& id )
		{
			if ( exists( id ) )
				return this->template create< T >( id );
			else
			{
				if ( this->load_resource( id ) )
				{
					return this->template create< T >( id );
				}
			}
			// NV_ASSERT( false, "resource_manager.get failed!" );
			return resource_type();
		}

		resource_type get( uint64 id )
		{
			if ( exists( shash64( id ) ) ) return this->template create< T >( shash64( id ) );
			// NV_ASSERT( false, "resource_manager.get failed!" );
			return resource_type();
		}

		virtual void clear()
		{
			for ( auto data : m_store )
			{
				release( data.second );
				policy_type::free( data.second );
			}
			m_store.clear();
		}

		virtual ~custom_resource_manager()
		{
			clear();
		}
	protected:

		resource_type add( shash64 id, stored_type resource )
		{
			auto m = m_store.find( shash64( id ) );
			if ( m != m_store.end() )
			{
				release( m->second );
			}
			m_store[id] = resource;
			return this->template create< T >( id );
		}

		virtual bool exists( resource_id id )
		{
			return m_store.find( shash64( id ) ) != m_store.end();
		}

		virtual void raw_add( resource_id id, void* value )
		{
			add( id, policy_type::to_stored( value ) );
		}


		virtual const void* raw_lock( resource_id id, bool )
		{
			auto m = m_store.find( id );
			return m != m_store.end() ? policy_type::to_pointer( m->second ) : nullptr;
		}

		virtual void unlock( resource_id ) {}
		virtual void release( resource_id ) {}
		virtual void release( stored_type ) {}

		hash_store< shash64, stored_type > m_store;
	};

	template < typename T, bool Heap = true >
	using lua_resource_manager = custom_resource_manager< T, Heap, lua_resource_manager_base >;


	template < typename T, bool Heap = true >
	class manual_resource_manager : public custom_resource_manager< T, Heap >
	{
	public:
		manual_resource_manager() {}
		using custom_resource_manager< T, Heap >::add;
	protected:
		virtual bool load_resource( const nv::string_view& ) { return false; }

	};
}

#endif // NV_ENGINE_RESOURCE_MANAGER_HH
