// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_SKELETON_INSTANCE_HH
#define NV_GFX_SKELETON_INSTANCE_HH

#include <nv/common.hh>
#include <nv/stl/array.hh>
#include <nv/interface/context.hh>
#include <nv/interface/mesh_data.hh>

namespace nv
{

	struct frame_range
	{
		uint32 start;
		uint32 end;
		bool   is_looping;

		constexpr frame_range() : start( 0 ), end( 0 ), is_looping( false ) {}
		constexpr frame_range( uint32 fstart, uint32 fend, bool fis_looping )
			: start( fstart )
			, end( fend )
			, is_looping( fis_looping )
		{
		}
		constexpr uint32 duration() const { return end - start; }

		inline float clamp_frame( float f ) const
		{
			if ( end <= start ) return float( start );
			float max = float( end - start );
			if ( f < max ) return f;
			return is_looping ? nv::fmodf( f, max ) : float( end );
		}

		inline float frame_from_time_fps( float time, uint32 fps ) const
		{
			return clamp_frame( time * float( fps ) );
		}

		inline float frame_from_ms_fps( uint32 ms, uint32 fps ) const
		{
			return clamp_frame( ( ms * 0.001f ) * float( fps ) );
		}

		inline float frame_from_ms_fps( uint32 ms, float fps ) const
		{
			return clamp_frame( ( ms * 0.001f ) * fps );
		}
	};

	template < typename Transform >
	class bone_transforms
	{
	public:
		typedef Transform value_type;

		bone_transforms() {}
		void prepare( const data_node_list& bone_data )
		{
			m_offsets.resize( bone_data.size() );
			for ( uint16 bi = 0; bi < bone_data.size(); ++bi )
				m_offsets[bi] = value_type( bone_data[bi].transform );
		}

		uint32 size() const { return m_offsets.size(); };
	protected:
		dynamic_array< value_type >   m_offsets;

		template < typename T >
		friend class skeleton_instance;
	};

	class skeleton_binding
	{
	public:
		skeleton_binding() {}
		void assign( const skeleton_binding& other );
		void prepare( const mesh_nodes_data* node_data, const data_node_list& bone_data );
		void prepare( const data_node_list& pose_data, const data_node_list& bone_data );
		uint32 size() const { return m_indices.size(); };
		uint32 skeleton_size() const { return m_bone_count; };
	protected:
		dynamic_array< sint16 > m_indices;
		data_descriptor         m_key;
		uint32                  m_bone_count;

		template < typename T >
		friend class skeleton_instance;
		friend class skeleton_transforms;
	};

	class skeleton_transforms
	{
	public:
		friend class pose_data_set;

		skeleton_transforms() {}
		const transform* xforms() const { return m_transforms.data(); }
		size_t size() const { return m_transforms.size(); }
		void assign( const data_node_list* node_data );
		void assign( const skeleton_transforms& other );
		void assign( const skeleton_transforms& other, const array_view< bool >& mask );

		void interpolate( const skeleton_transforms& a, const skeleton_transforms& b, float t, interpolation i );
		void interpolate( const skeleton_transforms& a, const skeleton_transforms& b, float t, interpolation i, const array_view< bool >& mask );
		void interpolate( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t, interpolation i );
		void interpolate( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t, interpolation i, const array_view< bool >& mask );
		void blend( const skeleton_transforms& a, const skeleton_transforms& b, float t, interpolation i, float blend, interpolation bi );
		void blend( const skeleton_transforms& a, const skeleton_transforms& b, float t, interpolation i, float blend, interpolation bi, const array_view< bool >& mask );
		void blend( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t, interpolation i, float blend, interpolation bi );
		void blend( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t, interpolation i, float blend, interpolation bi, const array_view< bool >& mask );

		void delocalize( const data_node_tree& node_data )
		{
			for ( uint32 n = 0; n < node_data.size(); ++n )
				if ( node_data[n].parent_id == -1 )
					delocalize_rec( node_data, n, transform() );
		}

		void delocalize_rec( const data_node_tree& node_data, uint32 id, const transform& parent );

//	protected:
		dynamic_array< transform > m_transforms;
	};

	template < typename Transform >
	class skeleton_instance
	{
	public:
		typedef Transform value_type;

		skeleton_instance() {}
		const mat4* xforms() const { return m_matrix.data(); }
		size_t size() const { return m_matrix.size(); }
		void assign( const skeleton_transforms& skeleton, const skeleton_binding& binding, const bone_transforms< value_type >& bones )
		{
			if ( bones.size() != m_matrix.size() ) m_matrix.resize( bones.size() );
			const transform* transforms = skeleton.xforms();
			for ( uint32 n = 0; n < skeleton.size(); ++n )
			{
				sint16 bone_id = binding.m_indices[n];
				if ( bone_id >= 0 )
				{
					// 			int too_complex;
					// 			transform tr( bones.m_offsets[bone_id] );
					// 			tr.set_orientation( normalize( tr.get_orientation() ) );
					// 			m_matrix[bone_id] = ( transforms[n] * tr ).extract();
					m_matrix[bone_id] = ( transforms[n] * bones.m_offsets[bone_id] ).extract();
				}
			}
		}
		void assign( const skeleton_transforms& skeleton, const bone_transforms< value_type >& bones )
		{
			if ( bones.size() != m_matrix.size() ) m_matrix.resize( bones.size() );
			const transform* transforms = skeleton.xforms();
			for ( uint32 n = 0; n < skeleton.size(); ++n )
			{
				m_matrix[n] = ( transforms[n] * bones.m_offsets[n]  ).extract();
			}
		}

		void assign( const bone_transforms< value_type >& bones )
		{
			if ( bones.size() != m_matrix.size() ) m_matrix.resize( bones.size() );
		}
	protected:

		dynamic_array< mat4 >      m_matrix;
	};


}

#endif // NV_GFX_SKELETON_INSTANCE_HH
