// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_POSES_HH
#define NV_GFX_POSES_HH

#include <nv/common.hh>
#include <nv/stl/vector.hh>
#include <nv/stl/array.hh>
#include <nv/stl/math.hh>
#include <nv/stl/string.hh>
#include <nv/stl/hash_store.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/interface/data_channel.hh>
#include <nv/interface/interpolation_raw.hh>
#include <nv/interface/interpolation_template.hh>
#include <nv/core/transform.hh>

namespace nv
{
	
	class pose_data_set
	{
	public:
		struct pose_set
		{
			uint32  start;
			uint32  count;
			shash64 name;
		};

		pose_data_set() : m_node_tree( shash64() ) {}
		uint32 size() const { return m_data.size(); }
		void initialize( const data_node_list& info )
		{
			NV_ASSERT_ALWAYS( m_node_tree.empty(), "Reinitialize!" );
			m_node_tree.assign( info );
			m_node_tree.initialize();
		}
		const data_node_tree& get_tree() const { return m_node_tree; }

		const skeleton_transforms& get( uint32 pose ) const
		{
			return m_data[pose];
		}


		uint32 add_poses( string_view id, const mesh_nodes_data& data )
		{
			uint32 nid = m_data.size();
			for ( int frame = 0; frame < data.get_frame_count(); frame++ )
			{
				add_pose( id, data, frame );
			}
			return nid;
		}

		uint32 add_pose( shash64 id, const mesh_nodes_data& data, int frame )
		{
			bool match = false;
			uint32 nid = m_data.size();

			auto& pset = m_sets[id];
			if ( !pset.name )
			{
				pset.start = nid;
				pset.count = 1;
				pset.name = id;
			}
			else
				pset.count++;

			m_data.push_back( skeleton_transforms() );
			auto& pose = m_data.back();

			if ( m_node_tree.empty() )
			{
				initialize( data.get_info() );
				match = true;
			}
			else
			{
				match = m_node_tree.matches( data.get_info() );
				//NV_ASSERT_ALWAYS( match, "mismatch!" );
			}

			pose.m_transforms.resize( m_node_tree.size() );

			for ( sint32 i = 0; i < sint32( m_node_tree.size() ); ++i )
			{
				const data_channel_set* node = nullptr;
				if ( match )
				{
					node = data[i];
				}
				else
				{
					node = data.get_by_hash( m_node_tree[i].name );
				}
				if ( node )
				{
					transform tf = raw_channel_interpolator( node ).get< transform >( float( frame ) );
					pose.m_transforms[i] = tf/*.get_orientation()*/;
				}
			}
			return nid;
		}


		data_node_tree m_node_tree;
		vector< skeleton_transforms > m_data;
		hash_store< shash64, pose_set > m_sets;
	};

}

#endif // NV_GFX_POSES_HH
