// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.
//
// TODO: 
//  * performance tests
//  * matching tests
//  * compatibility tests
//  * check if we can get rid of const templates
//
// String reference implementation based of string_ref proposal and boost::string_ref
//
// http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2012/n3442.html
// http://www.boost.org/doc/libs/1_58_0/libs/utility/doc/html/string_ref.html
//
// TODO: WARNING: this code is centered around 8bit char. In particular, 
//   const_string wont work with non-8bit chars.
// TODO: remove templating on char type, make a string_utf8 instead.

#ifndef NV_STL_STRING_HH
#define NV_STL_STRING_HH

#include <nv/stl/string/common.hh>
#include <nv/stl/string/string_base.hh>
#include <nv/stl/string/literal_string.hh>
#include <nv/stl/string/const_string.hh>
#include <nv/stl/string/short_string.hh>
#include <nv/stl/type_traits/primary.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/algorithm.hh>
#include <nv/stl/functional/hash.hh>

namespace nv
{
	
	namespace detail
	{
		// These could be done much better
		template <typename T>
		struct is_cstring_impl
		{
			typedef conditional_t < is_array<T>::value, remove_extent_t<T>*, remove_cv_t<T>	> decayed_type;
			typedef bool_constant <
				is_same<       char *, decayed_type >::value ||
				is_same< const char *, decayed_type >::value > type;
		};

	}


	template < typename T >
	struct is_cstring : detail::is_cstring_impl< remove_reference_t<T> >::type
	{

	};

	inline string_view trimmed( const string_view& str )
	{
		size_t endpos = str.find_last_not_of( " \r\n\t" );
		size_t startpos = str.find_first_not_of( " \r\n\t" );

		if ( string_view::npos != endpos || string_view::npos != startpos )
		{
			if ( string_view::npos == startpos ) startpos = 0;
			if ( string_view::npos != endpos )   endpos = endpos + 1 - startpos;
			return str.substr( startpos, endpos );
		}
		return str;
	}

	inline string_view rtrimmed( const string_view& str )
	{
		size_t endpos = str.find_last_not_of( " \r\n\t" );
		if ( string_view::npos != endpos )
		{
			return str.substr( 0, endpos + 1 );
		}
		return str;
	}

	inline string_view ltrimmed( const string_view& str )
	{
		size_t startpos = str.find_first_not_of( " \r\n\t" );
		if ( string_view::npos != startpos )
		{
			return str.substr( startpos );
		}
		return str;
	}


	inline string_view extract_extension( string_view filename )
	{
		size_t lastdot = filename.find_last_of( '.' );
		if ( string_view::npos != lastdot )
			return filename.substr( lastdot + 1 );
		return filename;
	}

	inline string_view without_extension( string_view filename )
	{
		size_t lastdot = filename.find_last_of( '.' );
		if ( string_view::npos != lastdot )
			return filename.substr( 0, lastdot );
		return filename;
	}

	struct string_hash_tag {};

	template < typename H >
	class string_hash : public hash_value< H, string_hash_tag >
	{
	public:
		typedef H hash_type;
		typedef H value_type;
		typedef string_hash< H > this_type;
		typedef hash_value< H, string_hash_tag > inherited_type;

		// TODO: enforce exact type?
		constexpr string_hash() : inherited_type() {}
		inline string_hash( const string_view& rhs ) : inherited_type( hash_string< H >( rhs.data(), rhs.size() ) ) {}
		template < typename S >
		inline string_hash( const string_base<S>& rhs ) : inherited_type( hash_string< H >( rhs.data(), rhs.size() ) ) {}
		constexpr string_hash( const char* str, size_t len ) : inherited_type( hash_string< H >( str, len ) ) {}
		constexpr string_hash( const char* str ) : inherited_type( hash_string< H >( str ) ) {}

		// Literal constructors
		template< size_t N >
		constexpr string_hash( char( &s )[N] ) : inherited_type( hash_string< H >( s, N - 1 ) ) {}
		template< size_t N >
		constexpr string_hash( const char( &s )[N] ) : inherited_type( hash_string< H >( s, N - 1 ) ) {}

		constexpr explicit string_hash( hash_type value ) : inherited_type( value ) {}
  		constexpr string_hash( const string_hash& value ) = default;
	};

	typedef string_hash< uint32 > shash32;
	typedef string_hash< uint64 > shash64;

	constexpr shash32 operator "" _sh32( const char* str, size_t len )
	{
		return shash32( detail::fnv_hash< uint32 >::hash( str, len ) );
	}

	constexpr shash64 operator "" _sh64( const char* str, size_t len )
	{
		return shash64( detail::fnv_hash< uint64 >::hash( str, len ) );
	}
}

using nv::operator "" _sh32;
using nv::operator "" _sh64;

#endif // NV_STL_STRING_HH
