// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_SKELETON_INSTANCE_HH
#define NV_GFX_SKELETON_INSTANCE_HH

#include <nv/common.hh>
#include <nv/stl/array.hh>
#include <nv/interface/context.hh>
#include <nv/interface/mesh_data.hh>

namespace nv
{

	struct frame_range
	{
		uint32 start;
		uint32 end;
		bool   is_looping;

		constexpr frame_range() : start( 0 ), end( 0 ), is_looping( false ) {}
		constexpr frame_range( uint32 fstart, uint32 fend, bool fis_looping )
			: start( fstart )
			, end( fend )
			, is_looping( fis_looping )
		{
		}
		constexpr uint32 duration() const { return end - start; }

		inline float clamp_frame( float f ) const
		{
			if ( end <= start ) return float( start );
			float max = float( end - start );
			if ( f < max ) return f;
			return is_looping ? nv::fmodf( f, max ) : float( end );
		}

		inline float frame_from_time_fps( float time, uint32 fps ) const
		{
			return clamp_frame( time * float( fps ) );
		}

		inline float frame_from_ms_fps( uint32 ms, uint32 fps ) const
		{
			return clamp_frame( ( ms * 0.001f ) * float( fps ) );
		}
	};

	class bone_transforms
	{
	public:
		bone_transforms() {}
		void prepare( const data_node_list& bone_data );
		uint32 size() const { return m_offsets.size(); };
	protected:
		dynamic_array< mat4 >   m_offsets;

		friend class skeleton_instance;
	};

	class skeleton_binding
	{
	public:
		skeleton_binding() {}
		void assign( const skeleton_binding& other );
		void prepare( const mesh_nodes_data* node_data, const data_node_list& bone_data );
		void prepare( const data_node_list& pose_data, const data_node_list& bone_data );
		uint32 size() const { return m_indices.size(); };
		uint32 skeleton_size() const { return m_bone_count; };
	protected:
		dynamic_array< sint16 > m_indices;
		data_descriptor         m_key;
		uint32                  m_bone_count;

		friend class skeleton_instance;
		friend class skeleton_transforms;
	};

	class skeleton_transforms
	{
	public:
		friend class pose_data_set;

		skeleton_transforms() {}
		const transform* xforms() const { return m_transforms.data(); }
		size_t size() const { return m_transforms.size(); }
		void assign( const data_node_list* node_data );
		void assign( const skeleton_transforms& other );

		void interpolate_linear( const skeleton_transforms& a, const skeleton_transforms& b, float t );
		void interpolate_nlerp( const skeleton_transforms& a, const skeleton_transforms& b, float t );
		void interpolate_slerp( const skeleton_transforms& a, const skeleton_transforms& b, float t );
		void interpolate4( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t );
		void interpolate_squad( const skeleton_transforms& s1, const skeleton_transforms& v1, const skeleton_transforms& v2, const skeleton_transforms& s2, float t );

// 		void animate( const mesh_nodes_data* node_data, float frame, bool local = false )
// 		{
// 			if ( local )
// 			{
// 				animate_local( node_data, frame );
// 				return;
// 			}
// 			if ( m_transforms.size() != node_data->size() )
// 				m_transforms.resize( node_data->size() );
// 			for ( uint32 n = 0; n < node_data->size(); ++n )
// 				if ( node_data->get_info( n ).parent_id == -1 )
// 					animate_rec( node_data, frame, n, transform(), false );
// 		}
		void blend_slerp( const skeleton_transforms& a, const skeleton_transforms& b, float t, float blend );

// 		void blend( const mesh_nodes_data* node_data, float frame, float blend, bool local = false )
// 		{
// 			NV_ASSERT( m_transforms.size() == node_data->size(), "skeleton size wrong!" );
// 			if ( local )
// 			{
// 				blend_local( node_data, frame, blend );
// 				return;
// 			}
// 			for ( uint32 n = 0; n < node_data->size(); ++n )
// 				if ( node_data->get_info( n ).parent_id == -1 )
// 					blend_rec( node_data, frame, n, transform(), false, blend );
// 		}

		void delocalize( const data_node_tree& node_data )
		{
			for ( uint32 n = 0; n < node_data.size(); ++n )
				if ( node_data[n].parent_id == -1 )
					delocalize_rec( node_data, n, transform() );
		}

		void delocalize_rec( const data_node_tree& node_data, uint32 id, const transform& parent );

//		void animate_rec( const mesh_nodes_data* node_data, float frame, uint32 id, const transform& parent, bool local );
// 		void blend_rec( const mesh_nodes_data* node_data, float frame, uint32 id, const transform& parent, bool local, float blend );
// 		void animate_local( const mesh_nodes_data* node_data, float frame );
// 		void blend_local( const mesh_nodes_data* node_data, float frame, float blend );

//	protected:
		dynamic_array< transform > m_transforms;
	};

	class skeleton_instance
	{
	public:
		skeleton_instance() {}
		const mat4* transforms() const { return m_matrix.data(); }
		size_t size() const { return m_matrix.size(); }
		void assign( const skeleton_transforms& skeleton, const skeleton_binding& binding, const bone_transforms& bones );
		void assign( const skeleton_transforms& skeleton, const bone_transforms& binding );
		void assign( const bone_transforms& binding );
	protected:

		dynamic_array< mat4 >      m_matrix;
	};


}

#endif // NV_GFX_SKELETON_INSTANCE_HH
