// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file animation.hh
 * @author Kornel Kisielewicz
 * @brief animation
 */

#ifndef NV_ENGINE_ANIMATION_HH
#define NV_ENGINE_ANIMATION_HH

#include <nv/common.hh>
#include <nv/core/resource.hh>
#include <nv/interface/mesh_data.hh>
#include <nv/gfx/skeleton_instance.hh>
#include <nv/gfx/poses.hh>

namespace nv
{
	class animator_bind_data;
	class animator_data;
}

NV_RTTI_DECLARE_NAME ( nv::mesh_nodes_data,    "nv::mesh_nodes_data"    )
NV_RTTI_DECLARE_NAME ( nv::animator_bind_data, "nv::animator_bind_data" )
NV_RTTI_DECLARE_NAME ( nv::animator_data,      "nv::animator_data"      )

namespace nv
{

	struct animator_clip_data
	{
		uint32      id;
		frame_range range;
	};

	class animator_data
	{
	public:
		animator_data( pose_data_set* data ) : m_data( data ) {}
		//
		pose_data_set* get_pose_data() { return m_data;  }
		const pose_data_set* get_pose_data() const { return m_data; }

		const animator_clip_data* get_clip( shash64 name ) const
		{
			auto it = m_clip_data.find( name );
			return ( it != m_clip_data.end() ? &it->second : nullptr );
		}
		bool has_clip( shash64 name ) const
		{
			auto it = m_clip_data.find( name );
			return it != m_clip_data.end();
		}
		void add_clip( shash64 name, animator_clip_data&& entry )
		{
			m_clip_data.assign( name, entry );
		}
	protected:
		pose_data_set* m_data;
		hash_store< shash64, animator_clip_data > m_clip_data;
	};

	// resource per animator/model set
	class animator_bind_data
	{
	public:
		animator_bind_data( const data_node_list& pose_frame, const data_node_list& bones )
			: m_bone_list( bones.get_name() )
		{
			m_bone_list.assign( bones );
			m_bone_transforms.prepare( bones );
			m_pose_binding.prepare( pose_frame, bones );
		}
		const bone_transforms& get_bone_transforms() const { return m_bone_transforms; }
		const skeleton_binding& get_pose_binding() const
		{
			return m_pose_binding;
		}
		const data_node_list& get_bone_list() const { return m_bone_list;  }

	protected:
		skeleton_binding                        m_pose_binding;
		bone_transforms                         m_bone_transforms;
		data_node_list                          m_bone_list;
	};

	// per model instance
	class animator_instance
	{
	public:
		animator_instance() {}
		void initialize( resource< animator_data > data, resource< animator_bind_data > bind_data )
		{
			m_data = data;
			m_bind_data = bind_data;
		}
		bool is_valid() const { return m_data.is_valid() && m_bind_data.is_valid(); }
		bool has_clip( shash64 id ) const
		{
			if ( auto data = m_data.lock() )
				return data->has_clip( id );
			else
				return false;
		}
		mat4 get_transform( uint32 idx ) const
		{
			if ( idx < m_skeleton.size() ) return m_skeleton.transforms()[idx];
			return mat4();
		}
		const skeleton_instance& get_skeleton() const
		{
			return m_skeleton;
		}

		sint16 get_bone_index( shash64 bone_name ) const
		{
			if ( is_valid() )
			{
				auto data = m_bind_data.lock();
				sint16 id = data->get_bone_list().resolve( bone_name );
				return id;
			}
			return -1;
		}

		const mat4& get_bone_matrix( shash64 bone_name ) const
		{
			if ( is_valid() )
			{
				auto data = m_bind_data.lock();
				sint16 id = data->get_bone_list().resolve( bone_name );
				if ( id >= 0 )
				{
					return data->get_bone_list()[id].transform;
				}
			}
			return mat4();
		}


		void update( shash64 clip, uint32 time )
		{
			if ( is_valid() )
			{
				auto data = m_data.lock();
				auto bind_data = m_bind_data.lock();
				const animator_clip_data* clip_data = data->get_clip( clip );

				if ( data->get_pose_data() )
				{
					if ( clip_data )
					{
						auto pose_data = data->get_pose_data();
						float fframe = clip_data->range.frame_from_ms_fps( time, 30 );

						unsigned v1 = unsigned( fframe );
						unsigned v2 = v1 + 1;
						if ( v2 >= clip_data->range.end ) v2 = clip_data->range.start;

						unsigned iv1 = v1 + clip_data->id;
						unsigned iv2 = v2 + clip_data->id;

						m_transforms.interpolate_slerp( pose_data->get( iv1 ), pose_data->get( iv2 ), fframe - v1 );
						m_transforms.delocalize( pose_data->get_tree() );

					}
					m_skeleton.assign( m_transforms, bind_data->get_pose_binding(), bind_data->get_bone_transforms() );
				}
			}
		}
	protected:
		resource< animator_data >      m_data;
		resource< animator_bind_data > m_bind_data;
		skeleton_transforms            m_transforms;
		skeleton_instance              m_skeleton;
	};


}

#endif // NV_ENGINE_ANIMATION_HH
