// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file terminal.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief resource interface
*/

#ifndef NV_CORE_RESOURCE_HH
#define NV_CORE_RESOURCE_HH

#include <nv/common.hh>
#include <nv/stl/string.hh>
#include <nv/stl/hash_store.hh>


namespace nv
{

	template < typename T > class resource;
	template < typename T > class resource_handle;
	template < typename T > class resource_lock;
	using resource_id = shash64;
	using resource_type_id = shash64;

	class resource_handler
	{

	protected:
		virtual const void* lock( resource_id id, resource_type_id type_hash ) = 0;
//		virtual const void* create( resource_id id, resource_type_id type_hash ) = 0;
		virtual void unlock( resource_id id, resource_type_id type_hash ) = 0;
		virtual void release( resource_id id, resource_type_id type_hash ) = 0;

		template< typename T >
		const T* lock( resource_id id )
		{
			return reinterpret_cast< const T* >( lock( id, shash64( rtti_type_hash<T>::hash() ) ) );
		}
		template< typename T >
		resource< T > create( resource_id id )
		{
			resource< T > result;
			result.m_id = id;
			result.m_handler = this;
			return result;
		}

// 		template< typename T >
// 		resource< T > create( resource_id id )
// 		{
// 			return reinterpret_cast<const T*>( create( id, rtti_type_hash<T>::hash() ) );
// 		}

		template < typename T >
		friend class resource;
		template < typename T >
		friend class resource_handle;
		template < typename T >
		friend class resource_lock;
//		friend class resource_manager;
	};


	template < typename T >
	class resource 
	{
	public:
		resource() : m_id(0), m_handler( nullptr ) {}
		resource_id id() const { return m_id; }
		constexpr bool is_valid() const { return m_id && m_handler; }
		constexpr explicit operator bool() const { return is_valid(); }
		~resource()
		{
			if ( m_handler ) m_handler->release( m_id, shash64( rtti_type_hash<T>::hash() ) );
		}
	protected:
		resource_id       m_id;
		resource_handler* m_handler;

		template < typename U >
		friend constexpr bool operator==( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator!=( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator<( const resource< U >&, const resource< U >& );
		template < typename U >
		friend constexpr bool operator>( const resource< U >&, const resource< U >& );

		friend class resource_lock< T >;
		friend class resource_handler;
	};

	template < typename T >
	constexpr bool operator== ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id == rhs.m_id && lhs.m_handler == rhs.m_handler;
	}

	template < typename T >
	constexpr bool operator!= ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id != rhs.m_id || lhs.m_handler != rhs.m_handler;
	}

	template < typename T >
	constexpr bool operator< ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id.value() < rhs.m_id.value();
	}

	template < typename T >
	constexpr bool operator> ( const resource< T >& lhs, const resource< T >& rhs )
	{
		return lhs.m_id.value() > rhs.m_id.value();
	}

	template < typename T >
	class resource_handle
	{
	public:

		void release( resource_handler* r )
		{
			r->release( m_id );
			m_id = 0;
		}
		~resource_handle()
		{
			NV_ASSERT( m_id == 0, "Resource not released!" );
		}
	protected:
		resource_id       m_id;

		friend class resource_lock< T >;
		friend class resource_handler;
	};

	template < typename T >
	class resource_lock
	{
		static const uint64 hash_value = rtti_type_hash<T>::value;
	public:
		explicit resource_lock( const resource< T >& r ) : m_id( r.m_id ), m_handler( r.m_handler ), m_resource( r.m_handler->lock<T>( r.m_id ) ) {}
		explicit resource_lock( const resource_handle< T >& r, resource_handler* handler ) : m_id( r.m_id ), m_handler( handler ), m_resource( handler->lock( r.m_id, shash64( hash_value ) ) ) {}
		const T& operator*() const { return *m_resource; }
		const T* operator->() const { return m_resource; }
		~resource_lock()
		{
			m_handler->unlock( m_id, shash64( hash_value ) );
		}
	private:
		resource_id       m_id;
		resource_handler* m_handler;
		const T*          m_resource;
	};


// 	class resource_manager : public resource_handler
// 	{
// 	public:
// 		template < typename T >
// 		shash64 register_resource_handler( resource_handler* handler )
// 		{
// 			return register_resource_handler( handler, rtti_type_hash<T>::hash() );
// 		}
// 
// 		~resource_manager()
// 		{
// 			for ( auto p : m_handlers )
// 			{
// 				delete p.second;
// 			}
// 		}
// 	protected:
// 		virtual const void* lock( resource_id id, resource_type_id type_hash )
// 		{
// 			auto handler = m_handlers.find( type_hash );
// 			NV_ASSERT( handler != m_handlers.end(), "Handler not registered!" );
// 			return handler->second->lock( id, type_hash );
// 		}
// 
// // 		virtual const void* create( resource_id id, resource_type_id type_hash )
// // 		{
// // 			auto handler = m_handlers.find( type_hash );
// // 			NV_ASSERT( handler != m_handlers.end(), "Handler not registered!" );
// // 			return handler->second->create( id, type_hash );
// // 		}
// 
// 		virtual void unlock( resource_id id, resource_type_id type_hash )
// 		{
// 			auto handler = m_handlers.find( type_hash );
// 			NV_ASSERT( handler != m_handlers.end(), "Handler not registered!" );
// 			handler->second->unlock( id, type_hash );
// 		}
// 
// 		virtual void release( resource_id id, resource_type_id type_hash )
// 		{
// 			auto handler = m_handlers.find( type_hash );
// 			NV_ASSERT( handler != m_handlers.end(), "Handler not registered!" );
// 			handler->second->release( id, type_hash );
// 		}
// 
// 		resource_type_id register_resource_handler( resource_handler* handler, resource_type_id type_hash )
// 		{
// 			NV_ASSERT( m_handlers.find( type_hash ) == m_handlers.end(), "Handler already registered!" );
// 			m_handlers[type_hash] = handler;
// 		}
// 
// 	protected:
// 		hash_store< resource_type_id, resource_handler* > m_handlers;
// 	};
}

#endif // NV_CORE_RESOURCE_HH

