// Copyright (C) 2011-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gfx/skeleton_instance.hh"

void nv::skeleton_instance::prepare( const mesh_nodes_data* bones )
{
	if ( m_offsets || m_indices ) return;
	hash_store< shash64, uint16 > bone_names;
	m_offsets = new mat4[bones->size()];
	m_indices = new sint16[m_data->size()];

	for ( nv::uint16 bi = 0; bi < bones->size(); ++bi )
	{
		const data_channel_set* bone = ( *bones )[bi];
		bone_names[bone->get_name()] = bi;
		m_offsets[bi] = bone->get_transform();
	}

	for ( uint32 n = 0; n < m_data->size(); ++n )
	{
		const data_channel_set* node = ( *m_data )[n];
		sint16 bone_id = -1;

		auto bi = bone_names.find( node->get_name() );
		if ( bi != bone_names.end() )
		{
			bone_id = sint16( bi->second );
		}
		m_indices[n] = bone_id;

		if ( m_key.size() == 0 && node->size() > 0 )
			m_key = node->get_interpolation_key();
	}
}

void nv::skeleton_instance::animate( mat4* data, float frame ) const
{
	if ( m_data->is_flat() )
	{
		animate_flat( data, frame );
	}
	else
	{
		for ( uint32 n = 0; n < m_data->size(); ++n )
			if ( ( *m_data )[n]->get_parent_id() == -1 )
				animate_rec( data, frame, n, mat4() );
	}
}

void nv::skeleton_instance::animate_flat( mat4* data, float frame ) const
{
	for ( uint32 n = 0; n < m_data->size(); ++n )
		if ( m_indices[n] >= 0 )
		{
			const data_channel_set* node = ( *m_data )[n];
			nv::mat4 node_mat( node->get_transform() );

			if ( node->size() > 0 )
			{
				raw_channel_interpolator interpolator( node, m_key );
				node_mat = interpolator.get< mat4 >( frame );
			}

			sint16 bone_id = m_indices[n];
			data[bone_id] = node_mat * m_offsets[bone_id];
		}
}

void nv::skeleton_instance::animate_rec( mat4* data, float frame, uint32 id, const mat4& parent ) const
{
	// TODO: fix transforms, which are now embedded,
	//       see note in assimp_loader.cc:load_node
	const data_channel_set* node = ( *m_data )[id];
	mat4 node_mat( node->get_transform() );

	if ( node->size() > 0 )
	{
		raw_channel_interpolator interpolator( node, m_key );
		node_mat = interpolator.get< mat4 >( frame );
	}

	mat4 global_mat = parent * node_mat;

	sint16 bone_id = m_indices[id];
	if ( bone_id >= 0 )
	{
		data[bone_id] = global_mat * m_offsets[bone_id];
	}

	for ( auto child : m_data->children( id ) )
	{
		animate_rec( data, frame, child, global_mat );
	}
}
