// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file resource_manager.hh
 * @author Kornel Kisielewicz
 * @brief engine resource manager
 */

#ifndef NV_ENGINE_RESOURCE_MANAGER_HH
#define NV_ENGINE_RESOURCE_MANAGER_HH

#include <nv/common.hh>
#include <nv/interface/context.hh>
#include <nv/core/resource.hh>
#include <nv/lua/lua_state.hh>
#include <nv/stl/hash_store.hh>
#include <nv/stl/vector.hh>

namespace nv
{

	typedef uint32 res_id;
	typedef uint32 res_type_id;

	class resource_system;

	class resource_manager_base : public resource_handler
	{
	public:	
		resource_manager_base() : m_lua( nullptr ) {}
		void initialize( lua::state* state );
		virtual string_view get_storage_name() const = 0;
		virtual string_view get_resource_name() const = 0;
		virtual void clear() { m_names.clear(); }
		void load_all();
		res_id load_resource( const string_view& id );
		virtual ~resource_manager_base() {}
	protected:
		virtual res_id load_resource( lua::table_guard& table ) = 0;

//		virtual const void* lock( resource_id id, resource_type_id );
		virtual void unlock( resource_id, resource_type_id ) {};
		virtual void release( resource_id, resource_type_id ) {};

		lua::state* m_lua;
		hash_store< shash64, res_id > m_names;
	};

	template < typename T >
	class resource_manager : public resource_manager_base
	{
	public:
		resource_manager() { m_data.push_back(nullptr); }
		T* get_resource( res_id id )
		{
			return ( id < m_data.size() ? m_data[ id ] : nullptr );
		}
		T* get_resource( const string_view& id )
		{
			auto m = m_names.find( id );
			if ( m != m_names.end() )
			{
				return get_resource( m->second );
			}
			return get_resource( load_resource( id ) );
		}
		resource< T > get( const string_view& id )
		{
			auto m = m_names.find( id );
			if ( m != m_names.end() )
			{
				return create< T >( id );
			}
			else
			{
				if ( get_resource( id ) != nullptr )
				{
					return create< T >( id );
				}
			}
//			NV_ASSERT( false, "resource_manager.get failed!" );
			return resource<T>();
		}
		virtual void clear()
		{
			resource_manager_base::clear();
			for ( uint32 i = 1; i < m_data.size(); ++i )
				release( m_data[i] );
			m_data.clear();
			m_data.push_back( nullptr );
		}

		virtual ~resource_manager() 
		{
			clear();
		}
	protected:
		virtual const void* lock( resource_id id, resource_type_id )
		{
			auto m = m_names.find( id );
			if ( m != m_names.end() )
			{
				return m_data[m->second];
			}
			return nullptr;
		}

		virtual void release( T* ) {}

		res_id add( T* resource )
		{
			m_data.push_back( resource );
			return m_data.size() - 1;
		}

		vector< T* > m_data;
	};
	
}

#endif // NV_ENGINE_RESOURCE_MANAGER_HH
