// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/gl/gl_device.hh"

#include "nv/gl/gl_window.hh"
#include "nv/core/logging.hh"
#include "nv/lib/sdl_image.hh"
#include "nv/gl/gl_enum.hh"
#include "nv/lib/gl.hh"

using namespace nv;

gl_device::gl_device()
{
	m_shader_header.append( "#version 330\n" );
	for ( auto& i : get_uniform_factory() ) 
		m_shader_header.append( "uniform "+datatype_to_glsl_type( i.second->get_datatype() )+" "+ i.first +";\n" );
	for ( auto& i : get_link_uniform_factory() ) 
		m_shader_header.append( "uniform sampler2D "+i.first +";\n" );
}

program gl_device::create_program( string_view vs_source, string_view fs_source )
{
	program result = m_programs.create();
	gl_program_info* info = m_programs.get( result );

	info->m_attribute_map   = new attribute_map;
	info->m_engine_uniforms = new engine_uniform_list;
	info->m_uniform_map     = new uniform_map;

	info->glid = glCreateProgram();
	compile( info, vs_source, fs_source );
	prepare_program( result );
	return result;
}

// this is a temporary function that will be removed once we find a way to 
// pass binary file data around
image_data* gl_device::create_image_data( string_view filename )
{
	load_sdl_image_library();
	SDL_Surface* image = IMG_Load( filename.data() );
	if (!image)
	{
		NV_LOG_ERROR( "Image file ", filename, " not found!" );
		return nullptr;
	}
	// TODO: BGR vs RGB, single channel
	pixel_format pformat = RGBA;
	switch ( image->format->BytesPerPixel )
	{
	case 4: pformat = RGBA; break;
	case 3: pformat = RGB; break;
	case 1: pformat = RED; break;
	default: NV_ASSERT( false, "BytesPerPixel != 4,3 or 1!" );
	}
	image_format format( pformat, UBYTE );
	image_data* data = new image_data( format, ivec2( image->w, image->h ), static_cast<nv::uint8*>( image->pixels ) );
	return data;
}

// this is a temporary function that will be removed once we find a way to 
// pass binary file data around
image_data* gl_device::create_image_data( const uint8* data, uint32 size )
{
	load_sdl_image_library();
	SDL_Surface* image = IMG_LoadTyped_RW( SDL_RWFromMem( const_cast<uint8*>( data ), int( size ) ), 1, "png" );
	if ( !image )
	{
		NV_LOG_ERROR( "Image binary data cannot be loaded found!" );
		return nullptr;
	}
	// TODO: BGR vs RGB, single channel
	NV_ASSERT( image->format->BytesPerPixel > 2, "bytes per pixel > 2!" );
	image_format format( image->format->BytesPerPixel == 3 ? RGB : RGBA, UBYTE );
	image_data* idata = new image_data( format, ivec2( image->w, image->h ), static_cast<nv::uint8*>( image->pixels ) );
	return idata;
}


gl_device::~gl_device()
{
	while ( m_textures.size() > 0 )
		release( m_textures.get_handle(0) );
	while ( m_buffers.size() > 0 )
		release( m_buffers.get_handle(0) );
	while ( m_programs.size() > 0 )
		release( m_programs.get_handle(0) );
}

nv::texture nv::gl_device::create_texture( texture_type type, ivec2 size, image_format aformat, sampler asampler, const void* data /*= nullptr */ )
{
	NV_ASSERT_ALWAYS( type != TEXTURE_3D && type != TEXTURE_2D_ARRAY, "2D texture type expected!" );
	unsigned glid = 0;
	unsigned gl_type = texture_type_to_enum( type );

	bool is_depth = aformat.format == DEPTH16 || aformat.format == DEPTH24 || aformat.format == DEPTH32;

	glGenTextures( 1, &glid );

	glBindTexture( gl_type, glid );

	// Detect if mipmapping was requested
	if ( gl_type == GL_TEXTURE_2D && asampler.filter_min != sampler::LINEAR && asampler.filter_min != sampler::NEAREST )
	{
		// TODO: This should not be done if we use framebuffers!
		glTexParameteri( gl_type, GL_GENERATE_MIPMAP, GL_TRUE);
	}

	if ( asampler.filter_max != sampler::NEAREST )
	{
		asampler.filter_max = sampler::LINEAR;
	}

	glTexParameteri( gl_type, GL_TEXTURE_MIN_FILTER, GLint( nv::sampler_filter_to_enum( asampler.filter_min ) ) );
	glTexParameteri( gl_type, GL_TEXTURE_MAG_FILTER, GLint( nv::sampler_filter_to_enum( asampler.filter_max ) ) );
	glTexParameteri( gl_type, GL_TEXTURE_WRAP_S, GLint( nv::sampler_wrap_to_enum( asampler.wrap_s) ) );
	glTexParameteri( gl_type, GL_TEXTURE_WRAP_T, GLint( nv::sampler_wrap_to_enum( asampler.wrap_t) ) );

	if ( is_depth )
	{
		glTexParameteri( gl_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
		glTexParameteri( gl_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

		// This is to allow usage of shadow2DProj function in the shader
		glTexParameteri( gl_type, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE );
		glTexParameteri( gl_type, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL );
	}

// #define GL_TEXTURE_MAX_ANISOTROPY_EXT 0x84FE
// #define GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT 0x84FF
// 
// 	float aniso = 0.0f;
// 	glGetFloatv( GL_MAX_TEXTURE_MAX_ANISOTROPY_EXT, &aniso );
// 	NV_LOG_INFO( "Anisotropy at ", aniso, " (", int( aniso ), " ) " );
// 	glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAX_ANISOTROPY_EXT, aniso );

	glTexImage2D( gl_type, 0, GLint( nv::image_format_to_internal_enum(aformat.format) ), size.x, size.y, 0, nv::image_format_to_enum(aformat.format), nv::datatype_to_gl_enum(aformat.type), data );

	glBindTexture( gl_type, 0 );

	texture result = m_textures.create();
	gl_texture_info* info = m_textures.get( result );
	info->type     = type;
	info->format   = aformat;
	info->tsampler = asampler;
	info->size     = ivec3( size.x, size.y, 1 );
	info->glid     = glid;
	return result;
}

nv::texture nv::gl_device::create_texture( texture_type type, ivec3 size, image_format aformat, sampler asampler, const void* data /*= nullptr */ )
{
	NV_ASSERT_ALWAYS( type == TEXTURE_3D || type == TEXTURE_2D_ARRAY, "3D texture type expected!" );
	unsigned glid = 0;
	unsigned gl_type = texture_type_to_enum( type );

	bool is_depth = aformat.format == DEPTH16 || aformat.format == DEPTH24 || aformat.format == DEPTH32;

	glGenTextures( 1, &glid );
	glBindTexture( gl_type, glid );

	if ( asampler.filter_max != sampler::NEAREST )
	{
		asampler.filter_max = sampler::LINEAR;
	}

	glTexParameteri( gl_type, GL_TEXTURE_MIN_FILTER, GLint( nv::sampler_filter_to_enum( asampler.filter_min ) ) );
	glTexParameteri( gl_type, GL_TEXTURE_MAG_FILTER, GLint( nv::sampler_filter_to_enum( asampler.filter_max ) ) );
	glTexParameteri( gl_type, GL_TEXTURE_WRAP_S, GLint( nv::sampler_wrap_to_enum( asampler.wrap_s ) ) );
	glTexParameteri( gl_type, GL_TEXTURE_WRAP_T, GLint( nv::sampler_wrap_to_enum( asampler.wrap_t ) ) );

	if ( is_depth )
	{
		glTexParameteri( gl_type, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
		glTexParameteri( gl_type, GL_TEXTURE_MAG_FILTER, GL_LINEAR );

		// This is to allow usage of shadow2DProj function in the shader
		glTexParameteri( gl_type, GL_TEXTURE_COMPARE_MODE, GL_COMPARE_REF_TO_TEXTURE );
		glTexParameteri( gl_type, GL_TEXTURE_COMPARE_FUNC, GL_LEQUAL );
	}

	//glTexStorage3D( GL_TEXTURE_2D_ARRAY, mipLevelCount, GL_RGBA8, width, height, layerCount );
	glTexImage3D( gl_type, 0, GLint( nv::image_format_to_internal_enum( aformat.format ) ), size.x, size.y, size.z, 0, nv::image_format_to_enum( aformat.format ), nv::datatype_to_gl_enum( aformat.type ), data );
	glBindTexture( gl_type, 0 );

	texture result = m_textures.create();
	gl_texture_info* info = m_textures.get( result );
	info->type = type;
	info->format = aformat;
	info->tsampler = asampler;
	info->size = size;
	info->glid = glid;
	return result;
}



void nv::gl_device::release( texture t )
{
	gl_texture_info* info = m_textures.get( t );
	if ( info )
	{
		if ( info->glid != 0 )
		{
			glDeleteTextures( 1, &(info->glid) );
		}
		m_textures.destroy( t );
	}
}

void nv::gl_device::release( buffer b )
{
	gl_buffer_info* info = m_buffers.get( b );
	if ( info )
	{
		if ( info->glid != 0 )
		{
			glDeleteBuffers( 1, &(info->glid) );
		}
		m_buffers.destroy( b );
	}
}

const texture_info* nv::gl_device::get_texture_info( texture t ) const
{
	return m_textures.get( t );
}

nv::buffer nv::gl_device::create_buffer( buffer_type type, buffer_hint hint, size_t size, const void* source /*= nullptr */ )
{
	unsigned glid   = 0;
	unsigned glenum = buffer_type_to_enum( type );
	glGenBuffers( 1, &glid );

	glBindBuffer( glenum, glid );
	glBufferData( glenum, GLsizeiptr( size ), source, buffer_hint_to_enum( hint ) );
	glBindBuffer( glenum, 0 );

	buffer result = m_buffers.create();
	gl_buffer_info* info = m_buffers.get( result );
	info->type = type;
	info->hint = hint;
	info->size = size;
	info->glid = glid;
	return result;
}

const buffer_info* nv::gl_device::get_buffer_info( buffer t ) const
{
	return m_buffers.get( t );
}

void nv::gl_device::release( program p )
{
	gl_program_info* info = m_programs.get( p );
	if ( info )
	{
		for ( auto& i : *info->m_uniform_map )
			delete i.second;

		glDetachShader( info->glid, info->glidv );
		glDetachShader( info->glid, info->glidf );
		glDeleteShader( info->glidv );
		glDeleteShader( info->glidf );
		glDeleteProgram( info->glid );

		delete info->m_attribute_map;
		delete info->m_engine_uniforms;
		delete info->m_uniform_map;

		m_programs.destroy( p );
	}
}

void nv::gl_device::prepare_program( program p )
{
	gl_program_info* info = m_programs.get( p );
	if ( info )
	{
		auto& map  = get_uniform_factory();
		auto& lmap = get_link_uniform_factory();

		for ( auto& i : *info->m_uniform_map )
		{
			auto j = lmap.find( i.first );
			if ( j != lmap.end() )
			{
				j->second->set( i.second );
			}			

			auto k = map.find( i.first );
			if ( k != map.end() )
			{
				info->m_engine_uniforms->push_back( k->second->create( i.second ) );
			}				
		}
	}
}

uniform_base* nv::gl_device::get_uniform( program p, const string_view& name, bool fatal /*= true */ ) const
{
	const gl_program_info* info = m_programs.get( p );
	{
		nv::uniform_map::const_iterator i = info->m_uniform_map->find( name );
		if ( i != info->m_uniform_map->end() )
		{
			return i->second;
		}
		if ( fatal )
		{
			NV_LOG_CRITICAL( "gl_device : uniform '", name, "' not found in program!" );
			NV_ABORT( "gl_device : uniform not found!" );
		}
	}
	return nullptr;
}

int nv::gl_device::get_attribute_location( program p, const string_view& name, bool fatal /*= true */ ) const
{
	const gl_program_info* info = m_programs.get( p );
	if ( info )
	{
		attribute_map::const_iterator i = info->m_attribute_map->find( name );
		if ( i != info->m_attribute_map->end() )
		{
			return i->second.location;
		}
		if ( fatal )
		{
			NV_LOG_CRITICAL( "gl_device : attribute '", name, "' not found in program!" );
			NV_ABORT( "gl_device : attribute not found!" );
		}
	}
	return -1;
}

int nv::gl_device::get_block_location( program p, const string_view& name, bool fatal /*= true */ ) const
{
	return -1;
}

bool nv::gl_device::compile( gl_program_info* p, string_view vertex_program, string_view fragment_program )
{
	if (!compile( GL_VERTEX_SHADER,   vertex_program, p->glidv ))   { return false; }
	if (!compile( GL_FRAGMENT_SHADER, fragment_program, p->glidf )) { return false; }

	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::POSITION   ), "nv_position"  );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::TEXCOORD   ), "nv_texcoord"  );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::NORMAL     ), "nv_normal"    );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::COLOR      ), "nv_color"     );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::TANGENT    ), "nv_tangent"   );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::BONEINDEX  ), "nv_boneindex" );
	glBindAttribLocation( p->glid, static_cast<GLuint>( slot::BONEWEIGHT ), "nv_boneweight");

	glAttachShader( p->glid, p->glidf );
	glAttachShader( p->glid, p->glidv );
	glLinkProgram( p->glid );

	const uint32 buffer_size = 2048;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int status;

	glGetProgramiv( p->glid, GL_LINK_STATUS, &status );
	glGetProgramInfoLog( p->glid, buffer_size, &length, buffer );

	NV_LOG_INFO( "Program #", p->glid, (status == GL_FALSE ? " failed to compile!" : " compiled successfully.") );

	if ( length > 0 )
	{
		NV_LOG_INFO( "Program #", p->glid, " log: ", string_view( buffer, size_t( length ) ) );
	}

	if ( status == GL_FALSE ) 
	{
		return false;
	}

	glValidateProgram( p->glid );
	glGetProgramiv( p->glid, GL_VALIDATE_STATUS, &status );

	if ( status == GL_FALSE )
	{
		glGetProgramInfoLog( p->glid, buffer_size, &length, buffer );
		NV_LOG_ERROR( "Program #", p->glid, " validation error : ", buffer );
		return false;
	}
	load_attributes( p );
	load_uniforms( p );
	return true;
}

void nv::gl_device::update_uniforms( gl_program_info* p )
{
	for ( auto& i : *p->m_uniform_map )
	{
		uniform_base* ubase = i.second;
		if ( ubase->is_dirty() )
		{
			int uloc = ubase->get_location();
			switch( ubase->get_type() )
			{
			case FLOAT          : glUniform1fv( uloc, ubase->get_length(), static_cast< uniform< enum_to_type< FLOAT >::type >*>( ubase )->get_value() ); break;
			case INT            : glUniform1iv( uloc, ubase->get_length(), static_cast< uniform< enum_to_type< INT >::type >*>( ubase )->get_value() ); break;
			case FLOAT_VECTOR_2 : glUniform2fv( uloc, ubase->get_length(), reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_VECTOR_2 >::type >*>( ubase )->get_value())); break;
			case FLOAT_VECTOR_3 : glUniform3fv( uloc, ubase->get_length(), reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_VECTOR_3 >::type >*>( ubase )->get_value())); break;
			case FLOAT_VECTOR_4 : glUniform4fv( uloc, ubase->get_length(), reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_VECTOR_4 >::type >*>( ubase )->get_value())); break;
			case INT_VECTOR_2   : glUniform2iv( uloc, ubase->get_length(), reinterpret_cast<const GLint*>( static_cast< uniform< enum_to_type< INT_VECTOR_2 >::type >*>( ubase )->get_value())); break;
			case INT_VECTOR_3   : glUniform3iv( uloc, ubase->get_length(), reinterpret_cast<const GLint*>( static_cast< uniform< enum_to_type< INT_VECTOR_3 >::type >*>( ubase )->get_value())); break;
			case INT_VECTOR_4   : glUniform4iv( uloc, ubase->get_length(), reinterpret_cast<const GLint*>( static_cast< uniform< enum_to_type< INT_VECTOR_4 >::type >*>( ubase )->get_value())); break;
			case FLOAT_MATRIX_2 : glUniformMatrix2fv( uloc, ubase->get_length(), GL_FALSE, reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_MATRIX_2 >::type >*>( ubase )->get_value())); break;
			case FLOAT_MATRIX_3 : glUniformMatrix3fv( uloc, ubase->get_length(), GL_FALSE, reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_MATRIX_3 >::type >*>( ubase )->get_value())); break;
			case FLOAT_MATRIX_4 : glUniformMatrix4fv( uloc, ubase->get_length(), GL_FALSE, reinterpret_cast<const GLfloat*>( static_cast< uniform< enum_to_type< FLOAT_MATRIX_4 >::type >*>( ubase )->get_value())); break;
			default : break; // error?
			}
			ubase->clean();
		}
	}
}

void nv::gl_device::load_attributes( gl_program_info* p )
{
	int params;
	glGetProgramiv( p->glid, GL_ACTIVE_ATTRIBUTES, &params );

	for ( unsigned i = 0; i < unsigned( params ); ++i )
	{
		int attr_nlen;
		int attr_len;
		unsigned attr_type;
		char name_buffer[128];

		glGetActiveAttrib( p->glid, i, 128, &attr_nlen, &attr_len, &attr_type, name_buffer );

		string_view name( name_buffer, size_t( attr_nlen ) );

		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int attr_loc = glGetAttribLocation( p->glid, name.data() );

		attribute& attr = (*p->m_attribute_map)[ name ];
		attr.location = attr_loc;
		attr.type     = gl_enum_to_datatype( attr_type );
		attr.length   = attr_len;
	}
}

void nv::gl_device::load_uniforms( gl_program_info* p )
{
	int params;
	int bparams;
	glGetProgramiv( p->glid, GL_ACTIVE_UNIFORMS, &params );
	glGetProgramiv( p->glid, GL_ACTIVE_UNIFORM_BLOCKS, &bparams );

	for ( unsigned i = 0; i < unsigned( params ); ++i )
	{
		int uni_nlen;
		int uni_len;
		unsigned uni_type;
		char name_buffer[128];

		glGetActiveUniform( p->glid, i, 128, &uni_nlen, &uni_len, &uni_type, name_buffer );

		string_view name( name_buffer, size_t( uni_nlen ) );

		// skip built-ins
		if ( name.substr(0,3) == "gl_" ) continue;

		int uni_loc = glGetUniformLocation( p->glid, name.data() );
		datatype utype = gl_enum_to_datatype( uni_type );

		// check for array
		size_t arrchar = name.find( '[' );
		if ( arrchar != string_view::npos )
		{
			name = name.substr( 0, arrchar );
		}

		uniform_base* u = uniform_base::create( utype, uni_loc, uni_len );
		NV_ASSERT( u, "Unknown uniform type!" );
		(*p->m_uniform_map)[ name ] = u;
	}

	for ( unsigned i = 0; i < unsigned( bparams ); ++i )
	{
		int uni_len;
		char name_buffer[128];

		glGetActiveUniformBlockName( p->glid, i, 128, &uni_len, name_buffer );
		NV_LOG_INFO( string_view( name_buffer, size_t( uni_len ) ) );
	}
}

bool nv::gl_device::compile( uint32 sh_type, string_view shader_code, unsigned& glid )
{
	glid = glCreateShader( sh_type );

	const char* pc = shader_code.data();
	int l = int( shader_code.length() );

	glShaderSource( glid, 1, &pc, &l );
	glCompileShader( glid );

	const uint32 buffer_size = 1024;
	char buffer[ buffer_size ] = { 0 };
	int length;
	int compile_ok = GL_FALSE;
	glGetShaderiv(glid, GL_COMPILE_STATUS, &compile_ok);
	glGetShaderInfoLog( glid, buffer_size, &length, buffer );

	if ( length > 0 )
	{
		if ( compile_ok == 0 )
		{
			NV_LOG_ERROR( "Shader #", glid, " error: ", string_view( buffer, size_t( length ) ) );
		}
		else
		{
			NV_LOG_INFO( "Shader #", glid, " compiled successfully: ", string_view( buffer, size_t( length ) ) );
		}
	}
	else
	{
		NV_LOG_INFO( "Shader #", glid, " compiled successfully." );
	}
	return compile_ok != 0;

}

