// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file image_data.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Image data struct
 */

#ifndef NV_INTERFACE_IMAGE_DATA_HH
#define NV_INTERFACE_IMAGE_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/algorithm/raw.hh>

namespace nv
{
	enum pixel_format
	{
		RGB,
		RGBA,
		RGB32F,
		RGBA32F,
		RGB16F,
		RGBA16F,
		BGR,
		BGRA,
		RED,
		DEPTH16,
		DEPTH24,
		DEPTH32,
	};
	
	struct image_format
	{
		pixel_format format;
		datatype     type;

		image_format( pixel_format f = RGB, datatype d = UBYTE ) 
			: format( f ), type( d )  {}
	};

	class image_data
	{
	public:
		image_data( image_format format, ivec2 size, const uint8 * data ) 
			: m_format( format ), m_size( size ), m_data( nullptr )	{ initialize( data ); }
		uint8* release_data() { uint8* r = m_data; m_data = nullptr; return r; }
		const uint8 * get_data()    const { return m_data; }
		const ivec2 get_size() const { return m_size; }
		// TODO : better depth check (template?)
		size_t get_depth() const { return m_format.format == RGB || m_format.format == BGR ? 3 : ( m_format.format == RED ? 1 : 4 ); }
		image_format get_format() const { return m_format; }
		~image_data() {	if (m_data) delete[] m_data; }
	private:
		void initialize( const uint8* data )
		{
			size_t bsize = static_cast<size_t>(m_size.x * m_size.y) * get_depth();
			m_data = new uint8[ bsize ]; 
			raw_copy( data, data + bsize, m_data );
		}

		image_format m_format;
		ivec2  m_size;  //!< Defines the size of the image as a vector
		uint8* m_data;  //!< Holder for data
	};
}

#endif // NV_INTERFACE_IMAGE_DATA_HH
