// Copyright (C) 2014-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_GFX_MESH_CREATOR_HH
#define NV_GFX_MESH_CREATOR_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/interface/mesh_data.hh>

namespace nv
{

	class mesh_data_creator
	{
	public:
		mesh_data_creator( data_channel_set* data );
		// assumes that position and normal is vec3, tangent is vec4
		void transform( float scale, const mat3& r33 );
		// TODO: this could generate normals too
		void generate_tangents();
		// assumes that position and normal is vec3, tangent is vec4
		void rotate_quadrant( uint8 rotation );
		void translate( vec3 offset );
		void flip_normals();
		void scale_texture( vec2 min, vec2 max );
		bool is_same_format( data_channel_set* other );
		void merge( data_channel_set* other );
	private:
		void initialize();

		raw_data_channel merge_channels( const raw_data_channel& a, const raw_data_channel& b );
		raw_data_channel append_channels( const raw_data_channel& a, const raw_data_channel& b, uint32 frame_count = 1 );

		raw_data_channel* m_pos_channel;
		raw_data_channel* m_nrm_channel;
		raw_data_channel* m_tan_channel;
		raw_data_channel* m_tex_channel;
		raw_data_channel* m_idx_channel;

		int m_pos_offset;
		int m_nrm_offset;
		int m_tan_offset;
		int m_tex_offset;
		int m_idx_offset;

		nv::datatype m_pos_type;
		nv::datatype m_nrm_type;
		nv::datatype m_tan_type;
		nv::datatype m_tex_type;
		nv::datatype m_idx_type;

		data_channel_set* m_data;
	};

	class mesh_nodes_creator
	{
	public:
		mesh_nodes_creator( mesh_nodes_data* data ) : m_data( data ) {}
		// assumes that keys are equally spaced
		void pre_transform_keys();
		// assumes that keys are equally spaced
		void merge_keys();
		void transform( float scale, const mat3& r33 );
	private:
		mesh_nodes_data* m_data;
	};

	class mesh_creator
	{
	public:
		mesh_creator( mesh_data_pack* pack ) : m_pack( pack ) {}
		void delete_mesh( uint32 index );
		// assumes that keys are equally spaced
		void pre_transform_keys() 
		{
			if ( m_pack->m_nodes )
				mesh_nodes_creator( m_pack->m_nodes ).pre_transform_keys();
		}
		void generate_tangents()
		{
			for ( size_t m = 0; m < m_pack->m_count; ++m )
				mesh_data_creator( &(m_pack->m_meshes[m]) ).generate_tangents();
		}

		// assumes that keys are equally spaced
		void merge_keys() { mesh_nodes_creator( m_pack->m_nodes ).merge_keys(); }
		// assumes that position and normal is vec3, tangent is vec4
		void flip_normals() 
		{
			for ( size_t m = 0; m < m_pack->m_count; ++m )
				mesh_data_creator( &(m_pack->m_meshes[m]) ).flip_normals();
		}
		void transform( float scale, const mat3& r33 ) 
		{
			for ( size_t m = 0; m < m_pack->m_count; ++m )
				mesh_data_creator( &(m_pack->m_meshes[m]) ).transform( scale, r33 );
			if ( m_pack->m_nodes )
				mesh_nodes_creator( m_pack->m_nodes ).transform( scale, r33 );
		}
		// assumes models have same format
		// currently only data merge
		bool merge( mesh_data_pack* other )
		{
			for ( size_t m = 0; m < m_pack->m_count; ++m )
			{
				if (!mesh_data_creator( &(m_pack->m_meshes[m]) ).is_same_format( &(other->m_meshes[m]) ) ) 
					return false;
			}
			for ( size_t m = 0; m < m_pack->m_count; ++m )
			{
				mesh_data_creator(&(m_pack->m_meshes[m]) ).merge( &(other->m_meshes[m]) );
			}
			return true;
		}
	private:
		mesh_data_pack* m_pack;
	};
}

#endif // NV_GFX_MESH_CREATOR_HH
