// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file quaternion.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief quaternion ops
*/

#ifndef NV_STL_MATH_QUATERNION_HH
#define NV_STL_MATH_QUATERNION_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/constants.hh>
#include <nv/stl/math/geometric.hh>
#include <nv/stl/math/angle.hh>

namespace nv
{

	namespace math
	{
		namespace detail
		{

			template < typename T >
			struct dot_impl< tquat< T > >
			{
				inline static T get( const tquat<T>& a, const tquat<T>& b )
				{
					tvec4<T> tmp( a.x * b.x, a.y * b.y, a.z * b.z, a.w * b.w );
					return ( tmp.x + tmp.y ) + ( tmp.z + tmp.w );
				}
			};

		}


		template < typename T >
		inline tquat<T> conjugate( const tquat<T>& q )
		{
			return tquat<T>( q.w, -q.x, -q.y, -q.z );
		}

		template < typename T >
		inline tquat<T> inverse( const tquat<T>& q )
		{
			return conjugate( q ) / dot( q, q );
		}

		template < typename T >
		inline T length( const tquat<T>& q )
		{
			return math::sqrt( dot( q, q ) );
		}

		template < typename T >
		inline tquat<T> normalize( const tquat<T>& q )
		{
			T len = length( q );
			if ( len <= T( 0 ) ) return tquat<T>( 1, 0, 0, 0 );
			T rlen = T( 1 ) / len;
			return tquat<T>( q.w * rlen, q.x * rlen, q.y * rlen, q.z * rlen );
		}

		template < typename T >
		inline tquat<T> cross( const tquat<T>& q1, const tquat<T>& q2 )
		{
			return tquat<T>(
				q1.w * q2.w - q1.x * q2.x - q1.y * q2.y - q1.z * q2.z,
				q1.w * q2.x + q1.x * q2.w + q1.y * q2.z - q1.z * q2.y,
				q1.w * q2.y + q1.y * q2.w + q1.z * q2.x - q1.x * q2.z,
				q1.w * q2.z + q1.z * q2.w + q1.x * q2.y - q1.y * q2.x );
		}

		template < typename T >
		inline tquat<T> mix( const tquat<T>& a, const tquat<T>& b, T m )
		{
			T cos_theta = dot( a, b );

			if ( cos_theta > T( 1 ) - epsilon<T>() )
			{
				return tquat<T>(
					mix( a.w, b.w, m ),
					mix( a.x, b.x, m ),
					mix( a.y, b.y, m ),
					mix( a.z, b.z, m ) );
			}
			else
			{
				T angle = acos( cos_theta );
				return ( sin( ( T( 1 ) - m ) * angle ) * a + sin( m * angle ) * b ) / sin( angle );
			}
		}

		template < typename T >
		inline tquat<T> lerp( const tquat<T>& a, const tquat<T>& b, T m )
		{
			NV_ASSERT( m >= static_cast<T>( 0 ), "Bad argument to lerp!" );
			NV_ASSERT( m <= static_cast<T>( 1 ), "Bad argument to lerp!" );

			return a * ( T( 1 ) - m ) + ( b * m );
		}

		template < typename T >
		inline tquat<T> slerp( const tquat<T>& x, const tquat<T>& y, T m )
		{
			tquat<T> z = y;
			T cos_theta = dot( x, y );
			if ( cos_theta < T( 0 ) )
			{
				z = -y;
				cos_theta = -cos_theta;
			}
			if ( cos_theta > T( 1 ) - epsilon<T>() )
			{
				return tquat<T>(
					mix( x.w, z.w, m ),
					mix( x.x, z.x, m ),
					mix( x.y, z.y, m ),
					mix( x.z, z.z, m ) );
			}
			else
			{
				T angle = acos( cos_theta );
				return ( sin( ( T( 1 ) - m ) * angle ) * x + sin( m * angle ) * z ) / sin( angle );
			}
		}

		template < typename T >
		inline tquat<T> rotate( const tquat<T>& q, T angle, const tvec3<T>& v )
		{
			tvec3<T> temp = v;
			T len = math::length( temp );
			if ( abs( len - T( 1 ) ) > T( 0.001 ) )
			{
				T rlen = static_cast<T>( 1 ) / len;
				temp.x *= rlen;
				temp.y *= rlen;
				temp.z *= rlen;
			}

			T sina = sin( angle * T( 0.5 ) );
			return q * tquat<T>( cos( angle * T( 0.5 ) ), temp.x * sina, temp.y * sina, temp.z * sina );
		}

		template < typename T >
		inline tvec3<T> euler_angles( const tquat<T>& x )
		{
			return tvec3<T>( pitch( x ), yaw( x ), roll( x ) );
		}

		template < typename T >
		inline T roll( const tquat<T>& q )
		{
			return T( atan( T( 2 ) * ( q.x * q.y + q.w * q.z ), q.w * q.w + q.x * q.x - q.y * q.y - q.z * q.z ) );
		}

		template < typename T >
		inline T pitch( const tquat<T>& q )
		{
			return T( atan( T( 2 ) * ( q.y * q.z + q.w * q.x ), q.w * q.w - q.x * q.x - q.y * q.y + q.z * q.z ) );
		}

		template < typename T >
		inline T yaw( const tquat<T>& q )
		{
			return asin( T( -2 ) * ( q.x * q.z - q.w * q.y ) );
		}

		template < typename T >
		inline tmat3<T> mat3_cast( const tquat<T>& q )
		{
			tmat3<T> result( T( 1 ) );
			T qxx( q.x * q.x );
			T qyy( q.y * q.y );
			T qzz( q.z * q.z );
			T qxz( q.x * q.z );
			T qxy( q.x * q.y );
			T qyz( q.y * q.z );
			T qwx( q.w * q.x );
			T qwy( q.w * q.y );
			T qwz( q.w * q.z );

			result[0][0] = 1 - 2 * ( qyy + qzz );
			result[0][1] = 2 * ( qxy + qwz );
			result[0][2] = 2 * ( qxz - qwy );

			result[1][0] = 2 * ( qxy - qwz );
			result[1][1] = 1 - 2 * ( qxx + qzz );
			result[1][2] = 2 * ( qyz + qwx );

			result[2][0] = 2 * ( qxz + qwy );
			result[2][1] = 2 * ( qyz - qwx );
			result[2][2] = 1 - 2 * ( qxx + qyy );
			return result;
		}

		template < typename T >
		inline tmat4<T> mat4_cast( const tquat<T>& q )
		{
			return tmat4<T>( mat3_cast( q ) );
		}

		template < typename T >
		inline tquat<T> quat_cast( const tmat3<T>& m )
		{
			T fx = m[0][0] - m[1][1] - m[2][2];
			T fy = m[1][1] - m[0][0] - m[2][2];
			T fz = m[2][2] - m[0][0] - m[1][1];
			T fw = m[0][0] + m[1][1] + m[2][2];

			int imax = 0;
			T fmax = fw;
			if ( fx > fmax )
			{
				fmax = fx;
				imax = 1;
			}
			if ( fy > fmax )
			{
				fmax = fy;
				imax = 2;
			}
			if ( fz > fmax )
			{
				fmax = fz;
				imax = 3;
			}

			T vmax = sqrt( fmax + T( 1 ) ) * T( 0.5 );
			T mult = static_cast<T>( 0.25 ) / vmax;

			tquat<T> result( ctor::uninitialize );
			switch ( imax )
			{
			case 0:
				result.w = vmax;
				result.x = ( m[1][2] - m[2][1] ) * mult;
				result.y = ( m[2][0] - m[0][2] ) * mult;
				result.z = ( m[0][1] - m[1][0] ) * mult;
				break;
			case 1:
				result.w = ( m[1][2] - m[2][1] ) * mult;
				result.x = vmax;
				result.y = ( m[0][1] + m[1][0] ) * mult;
				result.z = ( m[2][0] + m[0][2] ) * mult;
				break;
			case 2:
				result.w = ( m[2][0] - m[0][2] ) * mult;
				result.x = ( m[0][1] + m[1][0] ) * mult;
				result.y = vmax;
				result.z = ( m[1][2] + m[2][1] ) * mult;
				break;
			case 3:
				result.w = ( m[0][1] - m[1][0] ) * mult;
				result.x = ( m[2][0] + m[0][2] ) * mult;
				result.y = ( m[1][2] + m[2][1] ) * mult;
				result.z = vmax;
				break;

			default: break;
			}
			return result;
		}

		template < typename T >
		inline tquat<T> quat_cast( const tmat4<T>& m4 )
		{
			return quat_cast( tmat3<T>( m4 ) );
		}

		template < typename T >
		inline T angle( const tquat<T>& x )
		{
			return acos( x.w ) * T( 2 );
		}

		template < typename T >
		inline tvec3<T> axis( const tquat<T>& x )
		{
			T tmp1 = static_cast<T>( 1 ) - x.w * x.w;
			if ( tmp1 <= static_cast<T>( 0 ) ) return tvec3<T>( 0, 0, 1 );
			T tmp2 = static_cast<T>( 1 ) / sqrt( tmp1 );
			return tvec3<T>( x.x * tmp2, x.y * tmp2, x.z * tmp2 );
		}

		template < typename T >
		inline tquat<T> angle_axis( T angle, const tvec3<T>& v )
		{
			tquat<T> result( ctor::uninitialize );
			T sina = sin( angle * static_cast<T>( 0.5 ) );
			result.w = ( angle * static_cast<T>( 0.5 ) );
			result.x = v.x * sina;
			result.y = v.y * sina;
			result.z = v.z * sina;
			return result;
		}

		template < typename T >
		inline tvec4<bool> less_than( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] < lhs[i];
			return result;
		}

		template < typename T >
		inline tvec4<bool> less_than_equal( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] <= lhs[i];
			return result;
		}

		template < typename T >
		inline tvec4<bool> greater_than( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] > lhs[i];
			return result;
		}

		template < typename T >
		inline tvec4<bool> greater_than_equal( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool, P> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] >= lhs[i];
			return result;
		}

		template < typename T >
		inline tvec4<bool> equal( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool, P> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] == lhs[i];
			return result;
		}

		template < typename T >
		inline tvec4<bool> not_equal( const tquat<T>& lhs, const tquat<T>& rhs )
		{
			tvec4<bool> result( ctor::uninitialize );
			for ( component_count_t i = 0; i < component_count( x ); ++i )
				result[i] = rhs[i] != lhs[i];
			return result;
		}

	}

}

#endif // NV_STL_MATH_QUATERNION_HH
