// Copyright (C) 2015-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file common.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief transform matrices
*/

#ifndef NV_STL_MATH_MATRIX_TRANSFORM_HH
#define NV_STL_MATH_MATRIX_TRANSFORM_HH

#include <nv/stl/math/common.hh>
#include <nv/stl/math/constants.hh>

namespace nv
{
	namespace math
	{

		template < typename T >
		inline tmat4<T> translate( const tmat4<T>& m, const tvec3<T>& v )
		{
			tmat4<T> result( m );
			result[3] = m[0] * v[0] + m[1] * v[1] + m[2] * v[2] + m[3];
			return result;
		}

		template < typename T >
		inline tmat4<T> rotate( const tmat4<T>& m, T angle, const tvec3<T>& v )
		{
			T const a = angle;
			T const c = math::cos( a );
			T const s = math::sin( a );

			tvec3<T> axis( normalize( v ) );
			tvec3<T> temp( ( T( 1 ) - c ) * axis );

			tmat4<T> Rotate( ctor::uninitialize );
			Rotate[0][0] = c + temp[0] * axis[0];
			Rotate[0][1] = 0 + temp[0] * axis[1] + s * axis[2];
			Rotate[0][2] = 0 + temp[0] * axis[2] - s * axis[1];

			Rotate[1][0] = 0 + temp[1] * axis[0] - s * axis[2];
			Rotate[1][1] = c + temp[1] * axis[1];
			Rotate[1][2] = 0 + temp[1] * axis[2] + s * axis[0];

			Rotate[2][0] = 0 + temp[2] * axis[0] + s * axis[1];
			Rotate[2][1] = 0 + temp[2] * axis[1] - s * axis[0];
			Rotate[2][2] = c + temp[2] * axis[2];

			tmat4<T> result( ctor::uninitialize );
			result[0] = m[0] * Rotate[0][0] + m[1] * Rotate[0][1] + m[2] * Rotate[0][2];
			result[1] = m[0] * Rotate[1][0] + m[1] * Rotate[1][1] + m[2] * Rotate[1][2];
			result[2] = m[0] * Rotate[2][0] + m[1] * Rotate[2][1] + m[2] * Rotate[2][2];
			result[3] = m[3];
			return result;
		}

		template < typename T >
		inline tmat4<T> scale( const tmat4<T>& m, const tvec3<T>& v )
		{
			tmat4<T> result( ctor::uninitialize );
			result[0] = m[0] * v[0];
			result[1] = m[1] * v[1];
			result[2] = m[2] * v[2];
			result[3] = m[3];
			return result;
		}

		template < typename T >
		inline tmat4<T> translate( const tvec3<T>& v )
		{
			return translate( tmat4<T>( 1.0f ), v );
		}

		template < typename T >
		inline tmat4<T> rotate( T angle, const tvec3<T>& v )
		{
			return rotate( tmat4<T>( 1.0f ), angle, v );
		}

		template < typename T >
		inline tmat4<T> scale( const tvec3<T>& v )
		{
			return scale( tmat4<T>( 1.0f ), v );
		}

		template < typename T>
		inline tmat4<T> ortho( T left, T right, T bottom, T top, T z_near, T z_far )
		{
			tmat4<T> result( 1 );
			result[0][0] =  static_cast<T>( 2 ) / ( right - left );
			result[1][1] =  static_cast<T>( 2 ) / ( top - bottom );
			result[2][2] = -static_cast<T>( 2 ) / ( z_far - z_near );
			result[3][0] = -( right + left ) / ( right - left );
			result[3][1] = -( top + bottom ) / ( top - bottom );
			result[3][2] = -( z_far + z_near ) / ( z_far - z_near );
			return result;
		}

		template < typename T >
		inline tmat4<T> ortho( T left, T right, T bottom, T top )
		{
			tmat4<T> result( 1 );
			result[0][0] =  static_cast<T>( 2 ) / ( right - left );
			result[1][1] =  static_cast<T>( 2 ) / ( top - bottom );
			result[2][2] = -static_cast<T>( 1 );
			result[3][0] = -( right + left ) / ( right - left );
			result[3][1] = -( top + bottom ) / ( top - bottom );
			return result;
		}

		template < typename T >
		inline tmat4<T> frustum( T left, T right, T bottom, T top, T near_val, T far_val )
		{
			tmat4<T> result( 0 );
			result[0][0] = ( static_cast<T>( 2 ) * near_val ) / ( right - left );
			result[1][1] = ( static_cast<T>( 2 ) * near_val ) / ( top - bottom );
			result[2][0] = ( right + left ) / ( right - left );
			result[2][1] = ( top + bottom ) / ( top - bottom );
			result[2][2] = -( far_val + near_val ) / ( far_val - near_val );
			result[2][3] = static_cast<T>( -1 );
			result[3][2] = -( static_cast<T>( 2 ) * far_val * near_val ) / ( far_val - near_val );
			return result;
		}

		template < typename T >
		inline tmat4<T> perspective( T fovy, T aspect, T z_near, T z_far )
		{
			NV_ASSERT( abs( aspect - epsilon<T>() ) > static_cast<T>( 0 ), "bad parameters to math::perspective!" );

			const T tan_half_fovy = math::tan( fovy / static_cast<T>( 2 ) );

			tmat4<T> result( static_cast<T>( 0 ) );
			result[0][0] = static_cast<T>( 1 ) / ( aspect * tan_half_fovy );
			result[1][1] = static_cast<T>( 1 ) / ( tan_half_fovy );
			result[2][2] = -( z_far + z_near ) / ( z_far - z_near );
			result[2][3] = -static_cast<T>( 1 );
			result[3][2] = -( static_cast<T>( 2 ) * z_far * z_near ) / ( z_far - z_near );
			return result;
		}

		template < typename T >
		inline tmat4<T> perspective_fov( T fov, T width, T height, T z_near, T z_far )
		{
			NV_ASSERT( width > static_cast<T>( 0 ), "bad parameters to math::perspective_fov!" );
			NV_ASSERT( height > static_cast<T>( 0 ), "bad parameters to math::perspective_fov!" );
			NV_ASSERT( fov > static_cast<T>( 0 ), "bad parameters to math::perspective_fov!" );

			const T rad = fov;
			const T h = math::cos( static_cast<T>( 0.5 ) * rad ) / math::sin( static_cast<T>( 0.5 ) * rad );
			const T w = h * height / width; 

			tmat4<T> result( static_cast<T>( 0 ) );
			result[0][0] = w;
			result[1][1] = h;
			result[2][2] = -( z_far + z_near ) / ( z_far - z_near );
			result[2][3] = -static_cast<T>( 1 );
			result[3][2] = -( static_cast<T>( 2 ) * z_far * z_near ) / ( z_far - z_near );
			return result;
		}

		template < typename T >
		inline tmat4<T> infinite_perspective( T fovy, T aspect, T z_near, T ep = epsilon<T>() )
		{
			const T range  = math::tan( fovy / T( 2 ) ) * z_near;
			const T left   = -range * aspect;
			const T right  = range * aspect;
			const T bottom = -range;
			const T top    = range;

			tmat4<T> result( T( 0 ) );
			result[0][0] = ( static_cast<T>( 2 ) * zNear ) / ( right - left );
			result[1][1] = ( static_cast<T>( 2 ) * zNear ) / ( top - bottom );
			result[2][2] = ep - static_cast<T>( 1 );
			result[2][3] = static_cast<T>( -1 );
			result[3][2] = ( ep - static_cast<T>( 2 ) ) * z_near;
			return result;
		}

		template <typename T, typename U >
		inline tvec3<T> project( const tvec3<T>& obj, const tmat4<T>& model, const tmat4<T>& proj, const tvec4<U>& viewport )
		{
			tvec4<T> result = tvec4<T>( obj, T( 1 ) );
			result = model * result;
			result = proj * result;

			result /= result.w;
			result = result * T( 0.5 ) + T( 0.5 );
			result[0] = result[0] * T( viewport[2] ) + T( viewport[0] );
			result[1] = result[1] * T( viewport[3] ) + T( viewport[1] );

			return tvec3<T>( result );
		}

		template <typename T, typename U >
		inline tvec3<T> unproject( const tvec3<T>& win, const tmat4<T>& model, const tmat4<T>& proj, const tvec4<U>& viewport )
		{
			tmat4<T> invp = inverse( proj * model );

			tvec4<T> temp = tvec4<T>( win, T( 1 ) );
			temp.x = ( temp.x - T( viewport[0] ) ) / T( viewport[2] );
			temp.y = ( temp.y - T( viewport[1] ) ) / T( viewport[3] );
			temp = temp * T( 2 ) - T( 1 );

			tvec4<T> result = invp * temp;
			result /= result.w;
			return tvec3<T>( result );
		}

		template <typename T , typename U>
		inline tmat4<T> pick_matrix( const tvec2<T>& center, const tvec2<T>& delta, const tvec4<U>& viewport )
		{
			NV_ASSERT( delta.x > T( 0 ) && delta.y > T( 0 ), "bad delta passed to pick_matrix!" );
			if ( !( delta.x > T( 0 ) && delta.y > T( 0 ) ) ) return result;

			tmat4<T> result( 1 );
			tvec3<T> temp(
				( T( viewport[2] ) - T( 2 ) * ( center.x - T( viewport[0] ) ) ) / delta.x,
				( T( viewport[3] ) - T( 2 ) * ( center.y - T( viewport[1] ) ) ) / delta.y,
				T( 0 )
			);

			result = translate( result, temp );
			return scale( result, tvec3<T>( T( viewport[2] ) / delta.x, T( viewport[3] ) / delta.y, T( 1 ) ) );
		}

		template <typename T >
		inline tmat4<T> look_at( const tvec3<T>& eye, const tvec3<T>& center, const tvec3<T>& up )
		{
			const tvec3<T> f( normalize( center - eye ) );
			const tvec3<T> s( normalize( cross( f, up ) ) );
			const tvec3<T> u( cross( s, f ) );

			tmat4<T> result( 1 );
			result[0][0] = s.x;
			result[1][0] = s.y;
			result[2][0] = s.z;
			result[0][1] = u.x;
			result[1][1] = u.y;
			result[2][1] = u.z;
			result[0][2] = -f.x;
			result[1][2] = -f.y;
			result[2][2] = -f.z;
			result[3][0] = -dot( s, eye );
			result[3][1] = -dot( u, eye );
			result[3][2] = dot( f, eye );
			return result;
		}

	}
}

#endif // NV_STL_MATH_MATRIX_TRANSFORM_HH
