// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/io/c_stream.hh"
#include <cstdio>
#include <sys/stat.h> 

using namespace nv;

c_stream::c_stream()
	: m_file( nullptr )
	, m_file_name( "" )
	, m_file_size( size_t(-1) )
{

}

c_stream::c_stream( void* pfile, const char* filename )
	: m_file( pfile )
	, m_file_name( filename )
	, m_file_size( size_t(-1) )
{

}

c_stream::~c_stream()
{
	if ( m_file )
	{
		::fclose( reinterpret_cast<FILE*>( m_file ) );
	}
}

nv::size_t c_stream::read( void* buffer, nv::size_t size, nv::size_t count )
{
	NV_ASSERT( buffer != nullptr && size != 0 && count != 0, "Bad parameter passed to read!" );
	return m_file ? ::fread( buffer, size, count, reinterpret_cast<FILE*>( m_file ) ) : 0;
}

nv::size_t c_stream::write( const void* buffer, nv::size_t size, nv::size_t count )
{
	NV_ASSERT( buffer != nullptr && size != 0 && count != 0, "Bad parameter passed to write!" );
	return m_file ? ::fwrite( buffer, size, count, reinterpret_cast<FILE*>( m_file ) ) : 0;
}

bool c_stream::gets( char* buffer, size_t max_count )
{
	NV_ASSERT( buffer != nullptr && max_count != 0, "Bad parameter passed to write!" );
	char* result = ::fgets( buffer, max_count, reinterpret_cast<FILE*>( m_file ) );
	if ( !result ) return false;
	return true;
}

bool c_stream::seek( long offset, origin orig )
{
	return m_file != nullptr && ( ::fseek( reinterpret_cast<FILE*>( m_file ), offset, static_cast<int>(orig) ) == 0 );
}

nv::size_t c_stream::tell()
{
	return m_file != nullptr ? static_cast< nv::size_t >( ::ftell( reinterpret_cast<FILE*>( m_file ) ) ) : 0;
}

nv::size_t c_stream::size()
{
	if ( m_file == nullptr || m_file_name == nullptr )
	{
		return 0;
	}

	if ( m_file_size == size_t(-1) )
	{
		struct stat fstat; 
		int result = stat(m_file_name, &fstat ); 
		if ( result != 0 ) 
		{
			return 0; 
		}
		m_file_size = static_cast<size_t>(fstat.st_size); 		
	}

	return m_file_size;
}

void c_stream::flush()
{
	if ( m_file != nullptr )
	{
		::fflush( reinterpret_cast<FILE*>( m_file ) );
	}
}
