// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file fixed_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief fixed contiguous container handler
*/

#ifndef NV_STL_CONTAINER_FIXED_STORAGE_HH
#define NV_STL_CONTAINER_FIXED_STORAGE_HH

#include <nv/common.hh>
#include <nv/stl/container/initialize_policy.hh>

namespace nv
{

	template <
		typename Storage,
		size_t N,
		typename InitializePolicy = policy_initialize_standard
	>
	class fixed_storage : public Storage
	{
	public:
		typedef size_t                       size_type;
		typedef typename Storage::value_type value_type;

		static constexpr bool is_fixed = true;

		fixed_storage()
		{
			Storage::reallocate( N, false );
			InitializePolicy::initialize( Storage::data(), Storage::data() + N );
		}

		explicit fixed_storage( default_init )
		{
			Storage::reallocate( 0, false );
			uninitialized_construct( Storage::data(), Storage::data() + N );
		}

		explicit fixed_storage( const value_type& v )
		{
			uninitialized_fill( Storage::data(), Storage::data() + N, v );
		}

		static constexpr size_type max_size() { return N; }
		static constexpr size_type capacity() { return N; }
		static constexpr size_type size() { return N; }
		static constexpr bool empty() { return N == 0; }
		static constexpr size_type raw_size() { return sizeof( value_type ) * N; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), N ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), N ); }

		~fixed_storage()
		{
			InitializePolicy::destroy( Storage::data(), Storage::data() + N );
		}

		// prevent copying 
		fixed_storage( const fixed_storage& ) = delete;
		fixed_storage& operator=( const fixed_storage& ) = delete;
		// allow move
		fixed_storage( fixed_storage&& ) = default;
		fixed_storage& operator=( fixed_storage&& ) = default;
	};

}

#endif // NV_STL_CONTAINER_FIXED_STORAGE_HH
