// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file memory.hh
 * @author Kornel Kisielewicz
 * @brief memory utilities
 */

// TODO: implement const and mutable buffer using the union trick

#ifndef NV_STL_MEMORY_HH
#define NV_STL_MEMORY_HH

#include <nv/common.hh>
#include <nv/stl/type_traits/properties.hh>
#include <nv/stl/type_traits/alignment.hh>
#include <nv/stl/container/random_access.hh>
#include <nv/stl/algorithm/fill.hh>
#include <nv/stl/algorithm/copy.hh>
#include <nv/stl/utility.hh>
#include <nv/stl/iterator.hh>

namespace nv
{
	
	namespace mem_flags
	{
		static constexpr uint16 is_const  = 0x0001;
		static constexpr uint16 is_static = 0x0002;
		static constexpr uint16 read_only = 0x0004;
		static constexpr uint16 temporary = 0x0008;
	}

	template< typename T >
	class array_ref
	{
	public:
		typedef T                                    value_type;
		typedef size_t                               size_type;
		typedef ptrdiff_t                            difference_type;
		typedef value_type*                          pointer;
		typedef const value_type*                    const_pointer;
		typedef value_type&                          reference;
		typedef const value_type&                    const_reference;
		typedef value_type*                          iterator;
		typedef const value_type*                    const_iterator;
		typedef nv::reverse_iterator<iterator>       reverse_iterator;
		typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

		static constexpr bool is_static   = false;
		static constexpr bool is_fixed    = false;
		static constexpr bool is_const    = false;

		constexpr array_ref()
			: m_data( nullptr ), m_size( 0 ) {}
		constexpr array_ref( value_type* a_data, size_type a_size )
			: m_data( a_data ), m_size( a_size ) {}

		void assign( value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size != 0; }
		constexpr const value_type* data() const { return m_data; }
		inline    value_type* data() { return m_data; }
		constexpr size_type   raw_size() const { return sizeof( value_type ) * m_size; }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }
		inline    char* raw_data() { return reinterpret_cast<char*>( m_data ); }

		constexpr const_iterator begin() const { return const_iterator( m_data ); }
		constexpr const_iterator end() const { return const_iterator( m_data + m_size ); }
		inline    iterator begin() { return iterator( m_data ); }
		inline    iterator end() { return iterator( m_data + m_size ); }
		constexpr const_iterator cbegin() const { return const_iterator( m_data ); }
		constexpr const_iterator cend() const { return const_iterator( m_data + m_size ); }
		inline    reverse_iterator rbegin() { return reverse_iterator( end() ); }
		inline    reverse_iterator rend() { return reverse_iterator( begin() ); }
		constexpr const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
		constexpr const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
		constexpr const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
		constexpr const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }
		inline    const_iterator iat( size_t i ) const { NV_ASSERT_ALWAYS( i <= m_size, "Index out of range" ); return begin() + i; }
		inline    iterator       iat( size_t i ) { NV_ASSERT_ALWAYS( i <= m_size, "Index out of range" ); return begin() + i; }

		inline reference       front() { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" );  return m_data[0]; }
		inline const_reference front() const { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" ); return m_data[0]; }
		inline reference       back() { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }
		inline const_reference back() const { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }

		reference operator[]( size_type i ) { NV_ASSERT_ALWAYS( i < m_size, "Out of range" ); return m_data[i]; }
		const_reference operator[]( size_type i ) const { NV_ASSERT_ALWAYS( i < m_size, "Out of range" ); return m_data[i]; }
		inline void fill( const value_type& value ) { fill_n( iterator( m_data ), iterator( m_data + m_size ), value ); }

	protected:
		value_type* m_data;
		size_type   m_size;
	};

	template< typename T >
	class array_view
	{
	public:
		typedef T                                    value_type;
		typedef size_t                               size_type;
		typedef ptrdiff_t                            difference_type;
		typedef const value_type*                    pointer;
		typedef const value_type*                    const_pointer;
		typedef const value_type&                    reference;
		typedef const value_type&                    const_reference;
		typedef const value_type*                    iterator;
		typedef nv::reverse_iterator<iterator>       reverse_iterator;
		typedef const value_type*                    const_iterator;
		typedef nv::reverse_iterator<const_iterator> const_reverse_iterator;

		static constexpr bool is_static   = false;
		static constexpr bool is_fixed    = false;
		static constexpr bool is_const    = true;

		constexpr array_view()
			: m_data( nullptr ), m_size( 0 ) {}
		constexpr array_view( const value_type* a_data, size_type a_size )
			: m_data( a_data ), m_size( a_size ) {}
		constexpr array_view( const array_ref<T>& view )
			: m_data( view.data() ), m_size( view.size() ) {}

		void assign( const value_type* a_data, size_type a_size )
		{
			m_data = a_data;
			m_size = a_size;
		}

		constexpr size_type size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr const value_type* data() const { return m_data; }
		constexpr size_type   raw_size() const { return sizeof( value_type ) * m_size; }
		constexpr const char* raw_data() const { return reinterpret_cast<const char*>( m_data ); }

		constexpr const_iterator begin() const { return const_iterator( m_data ); }
		constexpr const_iterator end() const { return const_iterator( m_data + m_size ); }
		constexpr const_iterator cbegin() const { return const_iterator( m_data ); }
		constexpr const_iterator cend() const { return const_iterator( m_data + m_size ); }
		constexpr const_reverse_iterator rbegin() const { return const_reverse_iterator( end() ); }
		constexpr const_reverse_iterator crbegin() const { return const_reverse_iterator( end() ); }
		constexpr const_reverse_iterator rend() const { return const_reverse_iterator( begin() ); }
		constexpr const_reverse_iterator crend() const { return const_reverse_iterator( begin() ); }

		inline const_reference front() const { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" ); return m_data[0]; }
		inline const_reference back() const { NV_ASSERT_ALWAYS( !empty(), "front() called on empty data!" ); return m_data[m_size - 1]; }

		inline const_iterator iat( size_t i ) const { NV_ASSERT_ALWAYS( i <= m_size, "Index out of range" ); return begin() + i; }
		const_reference operator[]( size_type i ) const { NV_ASSERT_ALWAYS( i < m_size, "Out of range" ); return m_data[i]; }

	protected:
		const value_type* m_data;
		size_type         m_size;
	};

	template < typename T >
	inline void raw_construct_object( void* object )
	{
		new ( object )T;
	}

	template < typename T, typename ...Args >
	inline void raw_construct_object( void* object, Args&&... params )
	{
		new ( object )T( ::nv::forward<Args>( params )... );
	}

	template < typename T >
	inline void raw_move_construct_object( void* object, T&& original )
	{
		new ( object )T( ::nv::move( original ) );
	}

	template < typename T >
	inline void raw_copy_construct_object( void* object, const T& original )
	{
		new ( object )T( original );
	}

	template < typename T, typename ...Args >
	inline void construct_object( T* object, Args&&... params )
	{
		new ( object )T( ::nv::forward<Args>( params )... );
	}

	template < typename T >
	inline void move_construct_object( T* object, T&& original )
	{
		new ( object )T( ::nv::move( original ) );
	}

	template< typename T >
	inline void copy_construct_object( T* object, const T& original )
	{
		new ( object )T( original );
	}

	template < typename T >
	inline void destroy_object( T* object )
	{
		object->~T();
		NV_UNUSED( object ); // MSVC bug - if destructing basic type
	}

	template < typename T >
	void raw_destroy_object( void* object )
	{
		static_cast<T*>( object )->T::~T();
	}

	namespace detail
	{
		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, true_type )
		{
			fill( first, last, value );
		}

		template < typename ForwardIterator, typename T >
		void uninitialized_fill_impl( ForwardIterator first, ForwardIterator last, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( static_cast<void*>( addressof( *it ) ) ) value_type( value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, true_type )
		{
			return fill_n( first, count, value );
		}

		template < typename ForwardIterator, typename T >
		ForwardIterator uninitialized_fill_n_impl( ForwardIterator first, size_t count, const T& value, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new ( static_cast<void*>( addressof( *it ) ) ) value_type( value );
			return it;
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, true_type )
		{
			fill_default( first, last );
		}

		template < typename ForwardIterator >
		inline void uninitialized_construct_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				::new( static_cast<void*>( addressof( *it ) ) ) value_type;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return fill_default_n( first, count );
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_construct_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				::new( static_cast<void*>( addressof( *it ) ) ) value_type;
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator, ForwardIterator, true_type )
		{
			// no-op
		}

		template < typename ForwardIterator >
		inline void uninitialized_destroy_impl( ForwardIterator first, ForwardIterator last, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; it != last; ++it )
				( *it ).~value_type();
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, true_type )
		{
			return first + count;
		}

		template < typename ForwardIterator >
		inline ForwardIterator uninitialized_destroy_n_impl( ForwardIterator first, size_t count, false_type )
		{
			typedef typename iterator_traits< ForwardIterator >::value_type value_type;
			ForwardIterator it( first );
			for ( ; count > 0; --count, ++it )
				( *it ).~value_type();
			return it;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_impl( InputIterator first, InputIterator last, ForwardIterator out, false_type )
		{
			typedef typename iterator_traits<ForwardIterator>::value_type value_type;
			InputIterator src( first );
			ForwardIterator dest( out );
			for ( ; src != last; ++src, ++dest )
			{
				::new ( static_cast<void*>( addressof( *dest ) ) ) value_type( *src );
			}
			return dest;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_impl( InputIterator first, InputIterator last, ForwardIterator out, true_type )
		{
			return ForwardIterator( raw_alias_copy( first, last, out ) );
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_n_impl( InputIterator first, size_t count, ForwardIterator out, false_type )
		{
			typedef typename iterator_traits<ForwardIterator>::value_type value_type;
			InputIterator src( first );
			ForwardIterator dest( out );
			for ( ; count > 0; --count, ++src, ++dest )
			{
				::new ( static_cast<void*>( addressof( *dest ) ) ) value_type( *src );
			}
			return dest;
		}

		template < typename InputIterator, typename ForwardIterator >
		inline ForwardIterator uninitialized_copy_n_impl( InputIterator first, size_t count, ForwardIterator out, true_type )
		{
			return ForwardIterator( raw_alias_copy_n( out, first, count ) );
		}

	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill( ForwardIterator first, ForwardIterator last, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_fill_impl( first, last, value, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator, typename T >
	inline void uninitialized_fill_n( ForwardIterator first, size_t count, const T& value )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_fill_n_impl( first, count, value, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator, typename ...Args >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last, Args&&... params )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		ForwardIterator it( first );
		for ( ; it != last; ++it )
			::new( static_cast<void*>( addressof( *it ) ) ) value_type( forward<Args>( params )... );
	}

	template < typename ForwardIterator >
	inline void uninitialized_construct( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_construct_impl( first, last, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_construct_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_construct_n_impl( first, count, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline void uninitialized_destroy( ForwardIterator first, ForwardIterator last )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		detail::uninitialized_destroy_impl( first, last, has_trivial_destructor<value_type>() );
	}

	template < typename ForwardIterator >
	inline ForwardIterator uninitialized_destroy_n( ForwardIterator first, size_t count )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_destroy_n_impl( first, count, has_trivial_destructor<value_type>() );
	}

	template < typename InputIterator, typename ForwardIterator >
	inline ForwardIterator uninitialized_copy( InputIterator first, InputIterator last, ForwardIterator out )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_copy_impl( first, last, out, has_trivial_copy<value_type>() );
	}

	template < typename InputIterator, typename ForwardIterator >
	inline ForwardIterator uninitialized_copy_n( InputIterator first, size_t count, ForwardIterator out )
	{
		typedef typename iterator_traits< ForwardIterator >::value_type value_type;
		return detail::uninitialized_copy_n_impl( first, count, out, has_trivial_copy<value_type>() );
	}

}

#endif // NV_STL_MEMORY_HH
