// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file sized_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief sized contiguous container handler
*/

#ifndef NV_STL_CONTAINER_SIZED_STORAGE_HH
#define NV_STL_CONTAINER_SIZED_STORAGE_HH

#include <nv/common.hh>
#include <nv/stl/container/initialize_policy.hh>

namespace nv
{

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard
	>
	class sized_storage : public Storage
	{
	public:
		typedef size_t                       size_type;
		typedef typename Storage::value_type value_type;

		static constexpr bool is_fixed = false;

		static constexpr size_type max_size() { return size_type( 0x80000000 ); }
		constexpr size_t capacity() const { return m_size; }
		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr size_t raw_size() const { return sizeof( value_type ) * m_size; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), m_size ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), m_size ); }

		constexpr sized_storage() : m_size( 0 ) {}
		inline explicit sized_storage( size_type new_size ) : m_size( 0 ) { resize( new_size ); }
		inline explicit sized_storage( default_init ) : m_size( 0 ) { resize( default_init() ); }
		inline sized_storage( size_type new_size, const value_type& v ) : m_size( 0 ) { resize( new_size, v ); }

		// prevent copying 
		sized_storage( const sized_storage& ) = delete;
		sized_storage& operator=( const sized_storage& ) = delete;
		// allow move
		// allow move
		inline sized_storage( sized_storage&& other )
			: Storage( nv::move( other ) ), m_size( other.m_size )
		{
			other.m_size = 0;
		}
		inline sized_storage& operator=( sized_storage&& other )
		{
			if ( m_size > 0 ) Storage::reallocate( 0, false );
			m_size = other.m_size;
			Storage::operator=( nv::move( other ) );
			other.m_size = 0;
			return *this;
		}

		inline void resize( size_type new_size )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) InitializePolicy::initialize( Storage::data() + old_size, Storage::data() + m_size );
		}

		inline void resize( size_type new_size, default_init )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_construct( Storage::data() + old_size, Storage::data() + m_size );
		}

		inline void resize( size_type new_size, const value_type& value )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_fill( Storage::data() + old_size, Storage::data() + m_size, value );
		}

		inline void assign( const value_type* ptr, size_type sz )
		{
			if ( m_size > 0 ) InitializePolicy::destroy( Storage::data(), Storage::data() + m_size );
			if ( ptr != nullptr && sz > 0 )
			{
				if ( try_resize( sz, false ) )
					InitializePolicy::copy( ptr, ptr + sz, Storage::data() );
			}
			else try_resize( 0, false );
		}

		template< typename InputIterator >
		inline void assign( InputIterator first, InputIterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			size_type d = distance( first, last );
			if ( try_resize( d, false ) )
				InitializePolicy::copy( first, last, Storage::data() );
		}

		// explicit copy
		inline void assign( const sized_storage& other )
		{
			assign( other.data(), other.size() );
		}

		inline void clear()
		{
			if ( m_size > 0 )
			{
				InitializePolicy::destroy( Storage::data(), Storage::data() + m_size );
				try_resize( 0, false );
			}
		}

		~sized_storage()
		{
			clear();
		}

	protected:

		// TODO: return type error checking
		bool try_resize( size_t new_size, bool copy_needed )
		{
			if ( new_size != m_size )
			{
				if ( Storage::reallocate( new_size, copy_needed ) )
				{
					m_size = new_size;
					return true;
				}
				return false;
			}
			return true;
		}

		inline void resize_impl( size_type new_size )
		{
			size_type old_size = m_size;
			if ( new_size != old_size )
			{
				if ( new_size < old_size )
				{
					InitializePolicy::destroy( Storage::data() + new_size, Storage::data() + old_size );
				}
				if ( try_resize( new_size, true ) )
				{
					// TODO: error checking
				}
			}
		}
	protected:
		size_type m_size;
	};

}

#endif // NV_STL_CONTAINER_SIZED_STORAGE_HH
