// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file growing_storage.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief growing contiguous container handler
*/

#ifndef NV_STL_CONTAINER_GROWING_STORAGE_HH
#define NV_STL_CONTAINER_GROWING_STORAGE_HH

#include <nv/common.hh>
#include <nv/stl/container/contiguous_storage.hh>
#include <nv/stl/container/initialize_policy.hh>

namespace nv
{
	template< typename SizeType >
	struct default_next_capacity
	{
		static SizeType get( SizeType requested, SizeType capacity, SizeType max_size )
		{
			SizeType minimum = nv::max<SizeType>( capacity, 4 );
			SizeType remaining = max_size - capacity;
			if ( remaining < requested ) return 0;
			SizeType additional = nv::max( requested, capacity );
			return nv::max( minimum, remaining < additional ? max_size : capacity + additional );
		}
	};

	template <
		typename Storage,
		typename InitializePolicy = policy_initialize_standard,
		typename NextCapacity = default_next_capacity< size_t >
	>
	class growing_storage : public Storage
	{
	public:
		typedef typename Storage::value_type value_type;
		typedef size_t                       size_type;
		typedef value_type*                  iterator;
		typedef const value_type*            const_iterator;

		static constexpr bool is_fixed = false;

		static constexpr size_type max_size() { return size_type( 0x80000000 ); }
		constexpr size_t capacity() const { return m_capacity; }
		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr size_t raw_size() const { return sizeof( value_type ) * m_size; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), size() ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), size() ); }

		inline growing_storage() : m_size( 0 ), m_capacity( 0 ) {}
		inline explicit growing_storage( size_type new_size ) : m_size( 0 ), m_capacity( 0 ) { resize( new_size ); }
		inline explicit growing_storage( default_init ) : m_size( 0 ), m_capacity( 0 ) { resize( default_init() ); }
		inline growing_storage( size_type new_size, const value_type& v ) : m_size( 0 ), m_capacity( 0 ) { resize( new_size, v ); }

		// prevent copying 
		growing_storage( const growing_storage& ) = delete;
		growing_storage& operator=( const growing_storage& ) = delete;
		
		// allow move
		growing_storage( growing_storage&& other )
			: Storage( nv::move( other ) ), m_size( other.m_size ), m_capacity( other.m_capacity )
		{
			other.m_size = 0;
			other.m_capacity = 0;
		}

		inline growing_storage& operator=( growing_storage&& other )
		{
			if ( m_capacity > 0 ) Storage::reallocate( 0, false );
			m_size = other.m_size;
			m_capacity = other.m_capacity;
			Storage::operator=( nv::move( other ) );
			other.m_size = 0;
			other.m_capacity = 0;
			return *this;
		}

		void reserve( size_type new_capacity )
		{
			try_reserve( new_capacity, true );
		}

		void push_back( const value_type& e )
		{
			if ( try_grow( 1 ) ) copy_construct_object( Storage::data() + m_size - 1, e );
		}
		void push_back( value_type&& e )
		{
			if ( try_grow( 1 ) ) move_construct_object( Storage::data() + m_size - 1, forward<value_type>( e ) );
		}
		template < typename... Args >
		void emplace_back( Args&&... args )
		{
			if ( try_grow( 1 ) ) construct_object( Storage::data() + m_size - 1, forward<Args>( args )... );
		}

		void pop_back()
		{
			if ( m_size == 0 ) return;
			InitializePolicy::destroy( Storage::data() + m_size - 1 );
			m_size--;
		}

		// TODO: implement swap_erase
		iterator erase( iterator position )
		{
			if ( m_size == 0 ) return;
			iterator iend = Storage::data() + m_size;
			InitializePolicy::destroy( position );
			if ( ( position + 1 ) < iend )
				raw_alias_copy( position + 1, iend, position );
			m_size--;
			return position;
		}

		iterator erase( iterator first, iterator last )
		{
			iterator iend = Storage::data() + m_size;
			InitializePolicy::destroy( first, last );
			iterator position = raw_alias_copy( last, iend, first );
			m_size -= ( last - first );
			return position;
		}

		void append( const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			size_type d        = distance( first, last );
			size_type old_size = m_size;
			if ( try_grow( d ) )
				InitializePolicy::copy( first, last, Storage::data() + old_size );
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const value_type& value )
		{
			iterator iend = Storage::data() + m_size;
			if ( try_grow( 1 ) )
			{
				raw_alias_copy( position, iend, position + 1 );
				copy_construct_object( position, value );
			}
		}

		// TODO: This can be much optimized in the grow case by copying in three parts
		void insert( iterator position, const_iterator first, const_iterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			iterator iend = Storage::data() + m_size;
			size_type d = distance( first, last );
			if ( try_grow( d ) )
			{
				raw_alias_copy( position, iend, position + d );
				InitializePolicy::copy( first, last, position );
			}
		}

		inline void resize( size_type new_size )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) InitializePolicy::initialize( Storage::data() + old_size, Storage::data() + m_size );
		}

		inline void resize( size_type new_size, default_init )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_construct( Storage::data() + old_size, Storage::data() + m_size );
		}

		inline void resize( size_type new_size, const value_type& value )
		{
			size_type old_size = m_size;
			resize_impl( new_size );
			if ( m_size > old_size ) uninitialized_fill( Storage::data() + old_size, Storage::data() + m_size, value );
		}

		inline void assign( const value_type* ptr, size_type sz )
		{
			if ( ptr != nullptr && sz > 0 )
			{
				if ( try_resize( sz, false ) )
					InitializePolicy::copy( ptr, ptr + sz, Storage::data() );
			}
			else m_size = 0;
		}

		template< typename InputIterator >
		inline void assign( InputIterator first, InputIterator last )
		{
			// TODO: distance can't be called on destructive iterators - check 
			//   and use pushback if needed?
			if ( m_size > 0 ) InitializePolicy::destroy( Storage::data(), Storage::data() + m_size );
			size_type d = distance( first, last );
			if ( try_resize( d, false ) )
				InitializePolicy::copy( first, last, Storage::data() );
		}

		// explicit copy
		inline void assign( const growing_storage& other )
		{
			assign( other.data(), other.size() );
		}

		inline void clear()
		{
			if ( m_size > 0 )
			{
				InitializePolicy::destroy( Storage::data(), Storage::data() + m_size );
				m_size = 0;
			}
		}

		~growing_storage()
		{
			clear();
			Storage::reallocate( 0, false );
		}

	protected:

		inline void resize_impl( size_type new_size )
		{
			size_type old_size = m_size;
			if ( new_size != old_size )
			{
				if ( new_size < old_size )
				{
					InitializePolicy::destroy( Storage::data() + new_size, Storage::data() + old_size );
				}
				if ( try_resize( new_size, true ) )
				{
					// TODO: error checking
				}
			}
		}

		// TODO: return type error checking
		bool try_grow( size_t amount )
		{
			size_type new_size = amount + m_size;
			if ( new_size > m_capacity )
			{
				size_type new_capacity = NextCapacity::get( new_size - m_capacity, m_capacity, max_size() );
				if ( new_capacity > 0 && Storage::reallocate( new_capacity, true ) )
				{
					m_capacity = new_capacity;
				}
				else return false;
			}
			m_size = new_size;
			return true;
		}
		// TODO: return type error checking
		bool try_reserve( size_t new_capacity, bool copy_needed )
		{
			if ( new_capacity > m_capacity )
			{
				if ( new_capacity > 0 && Storage::reallocate( new_capacity, copy_needed ) )
				{
					m_capacity = new_capacity;
				}
				else return false;
			}
			return true;
		}
		// TODO: return type error checking
		bool try_resize( size_t new_size, bool copy_needed )
		{
			if ( new_size > m_size )
			{
				if ( try_reserve( new_size, copy_needed ) )
				{
					m_size = new_size;
					return true;
				}
				return false;
			}
			m_size = new_size;
			return true;
		}
	protected:
		size_type m_size;
		size_type m_capacity;

	};

}

#endif // NV_STL_CONTAINER_GROWING_STORAGE_HH

