// Copyright (C) 2012-2014 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_MESH_DATA_HH
#define NV_INTERFACE_MESH_DATA_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/string.hh>
#include <nv/interface/data_channel.hh>
#include <nv/gfx/animation.hh>

namespace nv
{

	struct index_u16 { uint16 index; };
	struct index_u32 { uint32 index; };

	using mesh_data = data_channel_set;

	struct mesh_node_data
	{
		std::string    name;
		sint16    target_id;
		sint16    parent_id;
		mat4      transform;
		data_channel_set* data;
	};

	class mesh_nodes_data
	{
		friend class mesh_creator;
		friend class mesh_nodes_creator;
	public:
		explicit mesh_nodes_data( const std::string& name, uint32 count, mesh_node_data* nodes )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(0), m_duration(0.0f), m_flat( false )
		{
		}

		explicit mesh_nodes_data( const std::string& name, uint32 count, mesh_node_data* nodes,
			uint16 a_fps, float a_frames, bool a_flat )
			: m_name( name ), m_count( count ), m_nodes( nodes ), m_frame_rate(a_fps), m_duration(a_frames), m_flat( a_flat )
		{
		}

		size_t get_count() const { return m_count; }

		const mesh_node_data* get_node( size_t i ) const 
		{
			if ( i >= m_count ) return nullptr;
			return &m_nodes[i];
		}

		const mesh_node_data* get_node( const std::string& name ) const 
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_nodes[ i ].name == name )
					return &m_nodes[ i ];
			}
			return nullptr;
		}

		int get_node_index( const std::string& name ) const 
		{
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_nodes[ i ].name == name )
					return int( i );
			}
			return -1;
		}

		data_channel_set* release_node_data( size_t i )
		{
			data_channel_set* result = m_nodes[i].data;
			m_nodes[i].data = nullptr;
			return result;
		}

		bool is_flat() const { return m_flat; }
		uint16 get_frame_rate() const { return m_frame_rate; }
		float get_duration() const { return m_duration; }
		void set_name( const std::string& name ) { m_name = name; }
		const std::string& get_name() const { return m_name; }

		~mesh_nodes_data()
		{
			if ( m_count > 0 )
				for ( uint32 i = 0; i < m_count; ++i ) delete m_nodes[i].data;
			delete[] m_nodes;
		}

	private:
		std::string m_name;
		uint32 m_count;
		mesh_node_data* m_nodes;
		uint16  m_frame_rate;
		float   m_duration;
		bool    m_flat;
	};

	class mesh_data_pack
	{
		friend class mesh_creator;
	public:
		explicit mesh_data_pack( uint32 a_count, mesh_data* a_meshes, mesh_nodes_data* a_nodes = nullptr )
		{
			m_count  = a_count;
			m_meshes = a_meshes;
			m_nodes  = a_nodes;
		}
		const mesh_data* get_mesh( uint32 index ) const
		{
			if ( index >= m_count ) return nullptr;
			return &m_meshes[ index ];
		}
// 		const mesh_data* get_mesh( const std::string& name ) const
// 		{
// 			for ( uint32 i = 0; i < m_count; ++i )
// 			{
// 				if ( m_meshes[ i ].get_name() == name )
// 					return &m_meshes[ i ];
// 			}
// 			return nullptr;
// 		}
		uint32 get_count() const { return m_count; }
		const mesh_nodes_data* get_nodes() const { return m_nodes; }
		~mesh_data_pack()
		{
			delete[] m_meshes;
			delete   m_nodes;
		}
	private:
		uint32           m_count;
		mesh_data*       m_meshes;
		mesh_nodes_data* m_nodes;
	};
}

#endif // NV_INTERFACE_MESH_DATA_HH
