// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file context.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief Context class
 */

#ifndef NV_INTERFACE_CONTEXT_HH
#define NV_INTERFACE_CONTEXT_HH

#include <nv/common.hh>
#include <nv/interface/device.hh>
#include <nv/interface/camera.hh>
#include <nv/interface/clear_state.hh>
#include <nv/interface/render_state.hh>

namespace nv
{
	struct vertex_array_tag {};
	struct framebuffer_tag {};
	typedef handle< uint32, 16, 16, vertex_array_tag > vertex_array;
	typedef handle< uint32, 16, 16, framebuffer_tag >  framebuffer;

	enum primitive
	{
		POINTS,
		LINES,
		LINE_LOOP,
		LINE_STRIP,
		TRIANGLES,
		TRIANGLE_STRIP,
		TRIANGLE_FAN
	};

	enum framebuffer_slot
	{
		READ_FRAMEBUFFER,
		DRAW_FRAMEBUFFER,
		FRAMEBUFFER,
	};

	class mesh_interface
	{
	public:
		mesh_interface() {}
		virtual void update( program ) {}
		virtual nv::vertex_array get_vertex_array() const = 0;
		virtual size_t get_index_count() const = 0;
		virtual nv::primitive get_primitive_type() const { return nv::TRIANGLES; }
		virtual ~mesh_interface() {}
	};

	struct vertex_array_info
	{
		static const int MAX_ATTRIBUTES = 24;

		uint32        count;
		buffer        index;
		bool          index_owner;
		datatype      index_type;
		vertex_buffer_attribute attr[MAX_ATTRIBUTES];
	};

	struct framebuffer_info
	{
		texture color_attachments[8];
		uint32  color_attachment_count;
		texture depth_attachment;
	};

	class context
	{
	public:
		context( device* a_device )	
		{ 
			m_device = a_device; 
		}

		virtual vertex_array create_vertex_array() = 0;
		virtual framebuffer create_framebuffer() = 0;
		virtual void release( vertex_array ) = 0;
		virtual void release( framebuffer ) = 0;

		virtual const vertex_array_info* get_vertex_array_info( vertex_array ) const = 0;
		virtual const framebuffer_info* get_framebuffer_info( framebuffer ) const = 0;

		virtual void set_draw_buffers( uint32 count, const output_slot* slots ) = 0;
		virtual void set_draw_buffer( output_slot slot ) = 0;
		virtual void set_read_buffer( output_slot slot ) = 0;
		virtual void blit( framebuffer f, clear_state::buffers_type mask, ivec2 src1, ivec2 src2, ivec2 dst1, ivec2 dst2 ) = 0;

		virtual void attach( framebuffer f, output_slot slot, texture t ) = 0;
		virtual void attach( framebuffer f, texture depth ) = 0;
		virtual void attach( framebuffer f, ivec2 size ) = 0;

		virtual bool check( framebuffer_slot ft ) = 0;
		virtual void bind( framebuffer f, framebuffer_slot slot = FRAMEBUFFER ) = 0;
		virtual void bind( texture, texture_slot ) = 0;
		virtual void update( texture, const void* ) = 0;
		virtual void update( buffer, const void*, size_t /*offset*/, size_t /*size*/ ) = 0;

		virtual void clear( const clear_state& cs ) = 0;
		// temporary
		virtual void draw( primitive prim, const render_state& rs, program p, vertex_array va, size_t count ) = 0;

		void draw( primitive prim, const render_state& rs, const scene_state& s, program p, vertex_array va, size_t count )
		{
			apply_engine_uniforms(p,s);
			draw( prim, rs, p, va, count );
		}
		virtual void apply_engine_uniforms( program p, const scene_state& s ) = 0;

		virtual void apply_render_state( const render_state& state ) = 0;
		virtual const ivec4& get_viewport() = 0;
		virtual void set_viewport( const ivec4& viewport ) = 0;
		virtual device* get_device() { return m_device; }
		virtual ~context() 
		{
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( vertex_array, buffer, const false_type& )
		{
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer_impl( vertex_array va, buffer vb, const true_type& )
		{
			typedef slot_info< VTX, SLOT > vinfo;
			typedef datatype_traits< typename vinfo::value_type > dt_traits;
			add_vertex_buffer( va, SLOT, vb, type_to_enum< typename dt_traits::base_type >::type, dt_traits::size, vinfo::offset, sizeof( VTX ), false );
		}

		template < typename VTX, slot SLOT >
		void add_vertex_buffer( vertex_array va, buffer vb )
		{
			add_vertex_buffer_impl< VTX, SLOT >( va, vb, has_slot< VTX, SLOT >() );
		}

		template < typename VTX >
		void add_vertex_buffers( vertex_array va, buffer vb )
		{
			add_vertex_buffer< VTX, slot::POSITION >  ( va, vb );
			add_vertex_buffer< VTX, slot::TEXCOORD >  ( va, vb );
			add_vertex_buffer< VTX, slot::NORMAL   >  ( va, vb );
			add_vertex_buffer< VTX, slot::TANGENT >   ( va, vb );
			add_vertex_buffer< VTX, slot::BONEINDEX > ( va, vb );
			add_vertex_buffer< VTX, slot::BONEWEIGHT >( va, vb );
			add_vertex_buffer< VTX, slot::COLOR >     ( va, vb );
		}

		void add_vertex_buffers( vertex_array va, buffer buf, const data_descriptor& descriptor )
		{
			for ( const auto& cslot : descriptor )
			{
				const datatype_info& info = get_datatype_info( cslot.etype );
				add_vertex_buffer( va, cslot.vslot, buf, info.base, info.elements, cslot.offset, descriptor.element_size(), false );
			}
		}


		template < typename VTXDATA >
		enable_if_t< is_class< VTXDATA >::value, vertex_array >
		create_vertex_array( const VTXDATA& data, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), hint );
		}

		template < typename VTXDATA, typename IDXDATA >
		enable_if_t< is_class< VTXDATA >::value && is_class< IDXDATA >::value, vertex_array >
		create_vertex_array( const VTXDATA& data, const IDXDATA& idata, buffer_hint hint )
		{
			return create_vertex_array( data.data(), data.size(), idata.data(), idata.size(), hint );
		}

		template < typename VTX >
		vertex_array create_vertex_array( const VTX* v, size_t count, buffer_hint hint )
		{
			// TODO: vb will not be owned or freed!
			vertex_array va = create_vertex_array();
			buffer       vb = m_device->create_buffer( VERTEX_BUFFER, hint, count * sizeof( VTX ), v );
			add_vertex_buffers< VTX >( va, vb );
			return va;
		}

		template < typename VTX, typename IDX >
		vertex_array create_vertex_array( const VTX* v, size_t vcount, const IDX* i, size_t icount, buffer_hint hint )
		{
			vertex_array va = create_vertex_array( v, vcount, hint );
			buffer       ib = m_device->create_buffer( INDEX_BUFFER, hint, icount * sizeof( IDX ), i );
			set_index_buffer( va, ib, type_to_enum< IDX >::type, true );
			return va;
		}

		void replace_vertex_buffer( vertex_array va, buffer vb, bool owner )
		{
			vertex_array_info* info = get_vertex_array_info_mutable( va );
			if ( info )
			{
				for ( uint32 i = 0; i < info->count; ++i )
				{
					vertex_buffer_attribute& vba = info->attr[i];
					if ( vba.owner ) m_device->release( vba.vbuffer );
					vba.vbuffer = vb;
					vba.owner   = owner;
				}
			}
		}

		void replace_vertex_buffer( vertex_array va, buffer vb, slot location, bool owner )
		{
			vertex_array_info* info = get_vertex_array_info_mutable( va );
			if ( info )
			{
				for ( uint32 i = 0; i < info->count; ++i )
				{
					if ( info->attr[i].location == location )
					{
						vertex_buffer_attribute& vba = info->attr[i];
						if ( vba.owner ) m_device->release( vba.vbuffer );
						vba.vbuffer = vb;
						vba.owner   = owner;
					}
				}
			}
		}

		void update_attribute_offset( vertex_array va, slot location, size_t offset ) 
		{
			vertex_array_info* info = get_vertex_array_info_mutable( va );
			if ( info )
			{
				for ( uint32 i = 0; i < info->count; ++i )
				{
					if ( info->attr[i].location == location )
					{
						info->attr[i].offset = offset;
					}
				}
			}
		}

		void set_index_buffer( vertex_array va, buffer b, datatype datatype, bool owner ) 
		{
			vertex_array_info* info = get_vertex_array_info_mutable( va );
			if ( info )
			{
				if ( m_device->get_buffer_info( b )->type == INDEX_BUFFER )
				{
					if (info->index.is_valid() && info->index_owner) m_device->release( info->index ); 

					info->index       = b; 
					info->index_owner = owner; 
					info->index_type  = datatype;
				}
			}
		}

		virtual void add_vertex_buffer( vertex_array va, slot location, buffer buf, datatype datatype, size_t components, size_t offset = 0, size_t stride = 0, bool owner = true )
		{
			vertex_array_info* info = get_vertex_array_info_mutable( va );
			if ( info )
			{
				NV_ASSERT( info->count < uint16( vertex_array_info::MAX_ATTRIBUTES ), "MAX_ATTRIBUTES reached!" );
				vertex_buffer_attribute& p = info->attr[ info->count ];
				p.vbuffer    = buf;
				p.dtype      = datatype;
				p.components = components;
				p.offset     = offset;
				p.stride     = stride;
				p.owner      = owner;
				p.location   = location;
				info->count++;
			}
		}

		buffer find_buffer( vertex_array va, slot location )
		{
			const vertex_array_info* info = get_vertex_array_info( va );
			if ( info )
			{
				for ( uint32 i = 0; i < info->count; ++i )
				{
					if ( info->attr[i].location == location )
					{
						return info->attr[i].vbuffer;
					}
				}
			}
			return buffer();
		}

		// TODO: HINTS ARE DIFFERENT!
		vertex_array create_vertex_array( const mesh_data* data, buffer_hint hint )
		{
			vertex_array  va = create_vertex_array();
			for ( auto channel : *data )
			{
				if ( channel->size() > 0 )
				{
					const data_descriptor& desc = channel->descriptor();
					// TODO: no if switch
					if ( desc[0].vslot == slot::INDEX )
					{
						buffer b = m_device->create_buffer( INDEX_BUFFER, hint, channel->raw_size(), channel->raw_data() );
						set_index_buffer( va, b, desc[0].etype, true );
					}
					else
					{
						buffer b = m_device->create_buffer( VERTEX_BUFFER, hint, channel->raw_size(), channel->raw_data() );
						add_vertex_buffers( va, b, desc );
					}
				}
			}
			return va;
		}

	protected:
		// TODO: remove
		virtual vertex_array_info* get_vertex_array_info_mutable( vertex_array ) = 0;

		device*      m_device;
		clear_state  m_clear_state;
		render_state m_render_state;
		ivec4        m_viewport;
	};

} // namespace nv

#endif // NV_INTERFACE_CONTEXT_HH
