// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_DATA_DESCRIPTOR_HH
#define NV_INTERFACE_DATA_DESCRIPTOR_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/type_traits/experimental.hh>

namespace nv
{

	enum class slot : uint8
	{
		POSITION    = 0,
		TEXCOORD    = 1,
		NORMAL      = 2,
		TANGENT     = 3,
		BONEINDEX   = 4,
		BONEWEIGHT  = 5,
		COLOR       = 6,

		INDEX       = 7,

		TIME        = 8,
		TRANSLATION = 9,
		ROTATION    = 10,
		SCALE       = 11,
		TFORM       = 12,

		MAX_STORE   = 8,
	};

	struct data_descriptor_slot
	{
		datatype etype  = NONE;
		uint32   offset = 0;
		slot     vslot  = slot::POSITION;
	};

	template < typename Struct, slot Slot, typename = void_t<> >
	struct has_slot : false_type {};

 	template < typename Struct >
 	struct has_slot< Struct, slot::POSITION, void_t< NV_VOID_DECLTYPE( Struct::position ) > > : true_type {};
	
	template < typename Struct >
	struct has_slot < Struct, slot::TEXCOORD, void_t< NV_VOID_DECLTYPE( Struct::texcoord ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::NORMAL, void_t< NV_VOID_DECLTYPE( Struct::normal ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TANGENT, void_t< NV_VOID_DECLTYPE( Struct::tangent ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::BONEINDEX, void_t< NV_VOID_DECLTYPE( Struct::boneindex ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::BONEWEIGHT, void_t< NV_VOID_DECLTYPE( Struct::boneweight ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::COLOR, void_t< NV_VOID_DECLTYPE( Struct::color ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::INDEX, void_t< NV_VOID_DECLTYPE( Struct::index ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TIME, void_t< NV_VOID_DECLTYPE( Struct::time ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TRANSLATION, void_t< NV_VOID_DECLTYPE( Struct::translation ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::ROTATION, void_t< NV_VOID_DECLTYPE( Struct::rotation ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::SCALE, void_t< NV_VOID_DECLTYPE( Struct::scale ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TFORM, void_t< NV_VOID_DECLTYPE( Struct::tform ) > > : true_type{};

	namespace detail
	{

		template < typename Struct, slot Slot, bool IsPresent >
		struct slot_info_impl
		{
		};

		template < typename Struct, slot Slot >
		struct slot_info_impl < Struct, Slot, false >
		{
			typedef empty_type value_type;
			static const datatype etype  = datatype::NONE;
			static const int      offset = 0;
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::POSITION, true >
		{
			typedef decltype( Struct::position ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::position ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, position );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TEXCOORD, true >
		{
			typedef decltype( Struct::texcoord ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::texcoord ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, texcoord );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::NORMAL, true >
		{
			typedef decltype( Struct::normal ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::normal ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, normal );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TANGENT, true >
		{
			typedef decltype( Struct::tangent ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::tangent ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, tangent );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::BONEINDEX, true >
		{
			typedef decltype( Struct::boneindex ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::boneindex ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, boneindex );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::BONEWEIGHT, true >
		{
			typedef decltype( Struct::boneweight ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::boneweight ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, boneweight );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::COLOR, true >
		{
			typedef decltype( Struct::color ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::color ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, color );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::INDEX, true >
		{
			typedef decltype( Struct::index ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::index ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, index );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TIME, true >
		{
			typedef decltype( Struct::time ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::time ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, time );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TRANSLATION, true >
		{
			typedef decltype( Struct::translation ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::translation ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, translation );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::ROTATION, true >
		{
			typedef decltype( Struct::rotation ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::rotation ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, rotation );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::SCALE, true >
		{
			typedef decltype( Struct::scale ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::scale ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, scale );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TFORM, true >
		{
			typedef decltype( Struct::tform ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::tform ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, tform );
		};
	}

	template < typename Struct, slot Slot >
	using slot_info = detail::slot_info_impl< Struct, Slot, has_slot< Struct, Slot >::value >;


	class data_descriptor
	{
	public:
		typedef const data_descriptor_slot* const_iterator;

		bool operator!=( const data_descriptor& rhs ) const
		{
			return !( *this == rhs );
		}

		bool operator==( const data_descriptor& rhs ) const
		{
			if ( m_size  != rhs.m_size )  return false;
			if ( m_count != rhs.m_count ) return false;
			for ( uint32 i = 0; i < m_count; ++i )
			{
				if ( m_slots[i].etype  != rhs.m_slots[i].etype )  return false;
				if ( m_slots[i].offset != rhs.m_slots[i].offset ) return false;
				if ( m_slots[i].vslot  != rhs.m_slots[i].vslot )  return false;
			}
			return true;
		}

		template < typename Struct >
		void initialize()
		{
			m_count = 0;
			initialize_slot< Struct, slot::POSITION >();
			initialize_slot< Struct, slot::TEXCOORD >();
			initialize_slot< Struct, slot::NORMAL >();
			initialize_slot< Struct, slot::TANGENT >();
			initialize_slot< Struct, slot::BONEINDEX >();
			initialize_slot< Struct, slot::BONEWEIGHT >();
			initialize_slot< Struct, slot::COLOR >();
			initialize_slot< Struct, slot::INDEX >();
			initialize_slot< Struct, slot::TIME >();
			initialize_slot< Struct, slot::TRANSLATION >();
			initialize_slot< Struct, slot::ROTATION >();
			initialize_slot< Struct, slot::SCALE >();
			initialize_slot< Struct, slot::TFORM >();
			m_size = sizeof( Struct );
		}

		uint32 element_size() const { return m_size; }
		uint32 slot_count() const { return m_count; }

		bool has_slot( slot vslot ) const
		{
			for ( const auto& dslot : *this )
				if ( dslot.vslot == vslot ) return true;
			return false;
		}

		const_iterator begin() const { return &m_slots[0]; }
		const_iterator end() const { return &m_slots[m_count]; }

		const data_descriptor_slot& operator []( uint32 i ) const
		{
			NV_ASSERT( i < m_count, "data_descriptor indexing failure!" );
			return m_slots[i];
		}

		void push_slot( datatype etype, slot vslot )
		{
			m_slots[m_count].etype  = etype;
			m_slots[m_count].offset = m_size;
			m_slots[m_count].vslot  = vslot;
			m_size += get_datatype_info( etype ).size;
			m_count++;
		}

		void append( const data_descriptor& desc )
		{
			for ( const auto& dslot : desc )
			{
				m_slots[m_count].etype  = dslot.etype;
				m_slots[m_count].offset = m_size;
				m_slots[m_count].vslot  = dslot.vslot;
				m_size += get_datatype_info( dslot.etype ).size;
				m_count++;
			}
		}

		template < typename Struct >
		static data_descriptor create()
		{
			data_descriptor result;
			result.initialize< Struct >();
			return result;
		}

	protected:

		data_descriptor_slot m_slots[uint16( slot::MAX_STORE )];
		uint32               m_count = 0;
		uint32               m_size = 0;

		template < typename Struct, slot Slot >
		void initialize_slot()
		{
			typedef slot_info< Struct, Slot > sinfo;
			m_slots[m_count].etype = sinfo::etype;
			if ( m_slots[m_count].etype != datatype::NONE )
			{
				m_slots[m_count].vslot  = Slot;
				m_slots[m_count].offset = sinfo::offset;
				m_count++;
			}
		}
	};

	struct raw_data_channel
	{
		raw_data_channel() : m_data( nullptr ), m_count( 0 ) {}
		~raw_data_channel()
		{
			if ( m_data != nullptr ) delete[] m_data;
		}

		const data_descriptor& descriptor() const { return m_desc;  }
		uint32 element_size() const { return m_desc.element_size(); }
		uint32 element_count() const { return m_count; }
		uint32 raw_size() const { return m_count * m_desc.element_size(); }
		const uint8* raw_data() const { return m_data; }

		// TODO: constexpr compile-time cast check?
		template < typename Struct >
		const Struct* data_cast() const 
		{ 
			return reinterpret_cast<const Struct*>( m_data );
		}

		bool has_slot( slot vslot ) const { return m_desc.has_slot( vslot ); }

		template < typename Struct >
		static raw_data_channel* create( uint32 count = 0 )
		{
			raw_data_channel* result = new raw_data_channel();
			result->m_desc.initialize<Struct>();
			result->m_count = count;
			result->m_data = ( count > 0 ? ( new uint8[result->raw_size()] ) : nullptr );
			return result;
		}
		static raw_data_channel* create( const data_descriptor& desc, uint32 count = 0 )
		{
			raw_data_channel* result = new raw_data_channel();
			result->m_desc = desc;
			result->m_count = count;
			result->m_data = ( count > 0 ? ( new uint8[result->raw_size()] ) : nullptr );
			return result;
		}

		template < typename Struct >
		friend class data_channel_creator;
		friend class raw_data_channel_creator;

	protected:
		uint8*          m_data;
		data_descriptor m_desc;
		uint32          m_count;

	};

	class raw_data_channel_creator
	{
	public:
		raw_data_channel_creator( const data_descriptor& desc, uint32 size )
		{
			m_channel = raw_data_channel::create( desc, size );
			m_owned   = true;
		}
		explicit raw_data_channel_creator( raw_data_channel* channel )
		{
			m_channel = channel;
			m_owned   = false;
		}

		uint32 element_size() const { return m_channel->element_size(); }
		uint32 size() const { return m_channel->element_count(); }
		uint32 raw_size() const { return m_channel->element_count() * m_channel->element_size(); }
		uint8* raw_data() { return m_channel->m_data; }
		const uint8* raw_data() const { return m_channel->m_data; }
		raw_data_channel* channel() { return m_channel; }

		data_descriptor& descriptor() { return m_channel->m_desc; }
		const data_descriptor& descriptor() const { return m_channel->m_desc; }

		raw_data_channel* release()
		{
			raw_data_channel* result = m_channel;
			m_channel = nullptr;
			return result;
		}

		~raw_data_channel_creator()
		{
			if ( m_owned && m_channel ) delete m_channel;
		}
	protected:
		raw_data_channel* m_channel;
		bool              m_owned;
	};

	template < typename Struct >
	class data_channel_creator : public raw_data_channel_creator
	{
	public:
		explicit data_channel_creator( uint32 size ) 
			: raw_data_channel_creator( data_descriptor::create< Struct >(), size ) {}
		// TODO - descriptor check
		explicit data_channel_creator( raw_data_channel* channel )
			: raw_data_channel_creator( channel ) {}
		
		Struct* data() { return reinterpret_cast< Struct* >( m_channel->m_data ); }
		const Struct* data() const { return reinterpret_cast< const Struct* >( m_channel->m_data ); }

		const Struct& operator []( uint32 i ) const
		{
			NV_ASSERT( i < m_channel->m_count, "data_channel_creator indexing failure!" );
			return reinterpret_cast< const Struct* >( m_channel->m_data )[i];
		}

		Struct& operator []( uint32 i ) 
		{
			NV_ASSERT( i < m_channel->m_count, "data_channel_creator indexing failure!" );
			return reinterpret_cast<Struct*>( m_channel->m_data )[i];
		}
	};
}

#endif // NV_INTERFACE_DATA_DESCRIPTOR_HH
