// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#ifndef NV_INTERFACE_DATA_DESCRIPTOR_HH
#define NV_INTERFACE_DATA_DESCRIPTOR_HH

#include <nv/common.hh>
#include <nv/stl/math.hh>
#include <nv/stl/type_traits/experimental.hh>

namespace nv
{

	enum class slot : uint8
	{
		POSITION    = 0,
		TEXCOORD    = 1,
		NORMAL      = 2,
		TANGENT     = 3,
		BONEINDEX   = 4,
		BONEWEIGHT  = 5,
		COLOR       = 6,

		INDEX       = 7,

		TIME        = 8,
		TRANSLATION = 9,
		ROTATION    = 10,
		SCALE       = 11,
		TFORM       = 12,

		MAX_STORE   = 8,
	};

	struct data_descriptor_slot
	{
		datatype etype  = NONE;
		uint32   offset = 0;
		slot     vslot  = slot::POSITION;
	};

	template < typename Struct, slot Slot, typename = void_t<> >
	struct has_slot : false_type {};

 	template < typename Struct >
 	struct has_slot< Struct, slot::POSITION, void_t< NV_VOID_DECLTYPE( Struct::position ) > > : true_type {};
	
	template < typename Struct >
	struct has_slot < Struct, slot::TEXCOORD, void_t< NV_VOID_DECLTYPE( Struct::texcoord ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::NORMAL, void_t< NV_VOID_DECLTYPE( Struct::normal ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TANGENT, void_t< NV_VOID_DECLTYPE( Struct::tangent ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::BONEINDEX, void_t< NV_VOID_DECLTYPE( Struct::boneindex ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::BONEWEIGHT, void_t< NV_VOID_DECLTYPE( Struct::boneweight ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::COLOR, void_t< NV_VOID_DECLTYPE( Struct::color ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::INDEX, void_t< NV_VOID_DECLTYPE( Struct::index ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TIME, void_t< NV_VOID_DECLTYPE( Struct::time ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TRANSLATION, void_t< NV_VOID_DECLTYPE( Struct::translation ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::ROTATION, void_t< NV_VOID_DECLTYPE( Struct::rotation ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::SCALE, void_t< NV_VOID_DECLTYPE( Struct::scale ) > > : true_type{};

	template < typename Struct >
	struct has_slot < Struct, slot::TFORM, void_t< NV_VOID_DECLTYPE( Struct::tform ) > > : true_type{};

	namespace detail
	{

		template < typename Struct, slot Slot, bool IsPresent >
		struct slot_info_impl
		{
		};

		template < typename Struct, slot Slot >
		struct slot_info_impl < Struct, Slot, false >
		{
			typedef empty_type value_type;
			static const datatype etype  = datatype::NONE;
			static const int      offset = 0;
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::POSITION, true >
		{
			typedef decltype( Struct::position ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::position ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, position );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TEXCOORD, true >
		{
			typedef decltype( Struct::texcoord ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::texcoord ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, texcoord );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::NORMAL, true >
		{
			typedef decltype( Struct::normal ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::normal ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, normal );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TANGENT, true >
		{
			typedef decltype( Struct::tangent ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::tangent ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, tangent );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::BONEINDEX, true >
		{
			typedef decltype( Struct::boneindex ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::boneindex ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, boneindex );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::BONEWEIGHT, true >
		{
			typedef decltype( Struct::boneweight ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::boneweight ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, boneweight );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::COLOR, true >
		{
			typedef decltype( Struct::color ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::color ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, color );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::INDEX, true >
		{
			typedef decltype( Struct::index ) value_type;
			static const datatype etype  = type_to_enum< decltype( Struct::index ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, index );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TIME, true >
		{
			typedef decltype( Struct::time ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::time ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, time );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TRANSLATION, true >
		{
			typedef decltype( Struct::translation ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::translation ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, translation );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::ROTATION, true >
		{
			typedef decltype( Struct::rotation ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::rotation ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, rotation );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::SCALE, true >
		{
			typedef decltype( Struct::scale ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::scale ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, scale );
		};

		template < typename Struct >
		struct slot_info_impl< Struct, slot::TFORM, true >
		{
			typedef decltype( Struct::tform ) value_type;
			static const datatype etype = type_to_enum< decltype( Struct::tform ) >::type;
			static const int      offset = NV_OFFSET_OF( Struct, tform );
		};
	}

	template < typename Struct, slot Slot >
	using slot_info = detail::slot_info_impl< Struct, Slot, has_slot< Struct, Slot >::value >;


	class data_descriptor
	{
	public:
		typedef const data_descriptor_slot* const_iterator;

		bool operator!=( const data_descriptor& rhs ) const
		{
			return !( *this == rhs );
		}

		bool operator==( const data_descriptor& rhs ) const
		{
//			if ( flags != rhs.flags ) return false;
			if ( size  != rhs.size )  return false;
			if ( count != rhs.count ) return false;
			for ( uint32 i = 0; i < count; ++i )
			{
				if ( slots[i].etype  != rhs.slots[i].etype )  return false;
				if ( slots[i].offset != rhs.slots[i].offset ) return false;
				if ( slots[i].vslot  != rhs.slots[i].vslot )  return false;
			}
			return true;
		}

		template < typename IndexType >
		void initialize_index()
		{
			count = 1;
			size = sizeof( IndexType );
			slots[0].etype = type_to_enum< IndexType >::type;
			slots[0].vslot = slot::INDEX;
			slots[0].offset = 0;
		}

		void initialize_index( datatype itype )
		{
			count = 1;
			size  = get_datatype_info( itype ).size;
			slots[0].etype  = itype;
			slots[0].vslot  = slot::INDEX;
			slots[0].offset = 0;
		}

		template < typename Struct >
		void initialize()
		{
			count = 0;
			initialize_slot< Struct, slot::POSITION >();
			initialize_slot< Struct, slot::TEXCOORD >();
			initialize_slot< Struct, slot::NORMAL >();
			initialize_slot< Struct, slot::TANGENT >();
			initialize_slot< Struct, slot::BONEINDEX >();
			initialize_slot< Struct, slot::BONEWEIGHT >();
			initialize_slot< Struct, slot::COLOR >();
			initialize_slot< Struct, slot::TIME >();
			initialize_slot< Struct, slot::TRANSLATION >();
			initialize_slot< Struct, slot::ROTATION >();
			initialize_slot< Struct, slot::SCALE >();
			initialize_slot< Struct, slot::TFORM >();
			size = sizeof( Struct );
		}

		uint32 element_size() const { return size; }
		uint32 slot_count() const { return count; }

		const_iterator begin() const { return &slots[0]; }
		const_iterator end() const { return &slots[count]; }

		const data_descriptor_slot& operator []( uint32 i ) const
		{
			NV_ASSERT( i < count, "data_descriptor indexing failure!" );
			return slots[i];
		}

		void push_slot( datatype etype, slot vslot )
		{
			slots[count].etype  = etype;
			slots[count].offset = size;
			slots[count].vslot  = vslot;
			size += get_datatype_info( etype ).size;
			count++;
		}

	protected:
		// 		enum flag_type : uint32
		// 		{
		// 			NONE   = 0b0000,
		// 			VERTEX = 0b0001,
		// 			INDEX  = 0b0010,
		// 			KEY    = 0b0100
		// 		};
		data_descriptor_slot slots[uint16( slot::MAX_STORE )];
		uint32               count = 0;
		uint32               size = 0;
		//		flag_type            flags = NONE;

		template < typename Struct, slot Slot >
		void initialize_slot()
		{
			typedef slot_info< Struct, Slot > sinfo;
			slots[count].etype = sinfo::etype;
			if ( slots[count].etype != datatype::NONE )
			{
				slots[count].vslot  = Slot;
				slots[count].offset = sinfo::offset;
				count++;
			}
		}
	};

}

#endif // NV_INTERFACE_DATA_DESCRIPTOR_HH

