// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.
//
// TODO: 
//  * performance tests
//  * matching tests
//  * compatibility tests
//  * check if we can get rid of const templates
//
// String reference implementation based of string_ref proposal and boost::string_ref
//
// http://www.open-std.org/jtc1/sc22/wg21/docs/papers/2012/n3442.html
// http://www.boost.org/doc/libs/1_58_0/libs/utility/doc/html/string_ref.html
//
// TODO: WARNING: this code is centered around 8bit char. In particular, 
//   const_string wont work with non-8bit chars.
// TODO: remove templating on char type, make a string_utf8 instead.

#ifndef NV_STL_STRING_HH
#define NV_STL_STRING_HH

#include <nv/common.hh>
#include <string>
#include <sstream>
#include <nv/stl/type_traits/primary.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/algorithm.hh>
#include <nv/stl/functional/hash.hh>

namespace nv
{


	// 	short_string< size_t >
	// 	string32
	// 	string64
	// 	string

	template<>
	struct hash< std::string, size_t >
	{
		static size_t get( const std::string& value )
		{
			return hash_string< size_t >( value.c_str(), value.length() );
		}
		inline size_t operator()( const std::string& value ) const { return get( value ); }
	};


	/**
	* Simple function for slurping a file into a string.
	*/
	std::string slurp( const std::string& filename );

	namespace detail
	{
		// These could be done much better
		template <typename T>
		struct is_cstring_impl
		{
			typedef conditional_t < is_array<T>::value, remove_extent_t<T>*, remove_cv_t<T>	> decayed_type;
			typedef bool_constant <
				is_same<       char *, decayed_type >::value ||
				is_same< const char *, decayed_type >::value > type;
		};


	}

	template < typename Storage > class string_base;
	class string_view;

	// Stronger version
	//	template < typename T >
	//	struct is_string_base : is_template_base_of< T, string_base > {};

	template < typename T, typename Enable = void >
	struct is_string_base : false_type {};
	template < typename T >
	struct is_string_base < T,
		typename enable_if<
		is_base_of< string_base< typename T::storage_type >, T >::value,
		void >::type > : true_type{};

	template < typename T >
	struct is_cstring : detail::is_cstring_impl< remove_reference_t<T> >::type
	{

	};

	// string base class - will become a base for a string class later
	template < typename Storage >
	class string_base : public Storage
	{
		typedef Storage                                     base_type;
		typedef string_base< base_type >                    this_type;
	public:
		typedef Storage                                     storage_type;
		typedef typename base_type::value_type              value_type;
		typedef typename base_type::pointer                 pointer;
		typedef typename base_type::const_pointer           const_pointer;
		typedef typename base_type::reference               reference;
		typedef typename base_type::const_reference         const_reference;
		typedef typename base_type::iterator                iterator;
		typedef typename base_type::const_iterator          const_iterator;
		typedef typename base_type::reverse_iterator        reverse_iterator;
		typedef typename base_type::const_reverse_iterator  const_reverse_iterator;
		typedef typename base_type::size_type               size_type;
		typedef typename base_type::difference_type         difference_type;

		static constexpr size_type npos = size_type( -1 );

		// conversion to std::string
		inline std::string to_string() const
		{
			return std::string( this->data(), this->size() );
		}

		inline size_type length()   const { return this->size(); }

		// string operations
		int compare( const string_view& rhs ) const;
		bool starts_with( value_type c ) const;
		bool starts_with( const string_view& s ) const;
		bool ends_with( value_type c ) const;
		bool ends_with( const string_view& s ) const;
		auto find( value_type c, size_type pos = 0 ) const->size_type;
		auto find( const string_view& s, size_type pos = 0 ) const->size_type;
		auto rfind( value_type c, size_type pos = 0 ) const->size_type;
		auto rfind( const string_view& s, size_type pos = 0 ) const->size_type;
		auto find_first_of( value_type c ) const->size_type;
		auto find_first_of( const string_view& s ) const->size_type;
		auto find_last_of( value_type c )  const->size_type;
		auto find_last_of( const string_view& s ) const->size_type;
		auto find_first_not_of( value_type c ) const->size_type;
		auto find_first_not_of( const string_view& s ) const->size_type;
		auto find_last_not_of( value_type c ) const->size_type;
		auto find_last_not_of( const string_view& s ) const->size_type;

		// string operations
		string_view substr( size_type p, size_type n = npos ) const;

		template < typename H = size_type >
		inline H get_hash() const
		{
			return hash_string< size_type >( this->data(), this->size() );
		}

	protected:
		using base_type::base_type;
		constexpr string_base() : base_type() {}
		constexpr string_base( pointer str, size_type len ) : base_type( str, len ) {}

		template < typename ReverseIterator >
		size_type reverse_distance( ReverseIterator first, ReverseIterator last ) const
		{
			return this->size() - 1 - static_cast<size_t>( nv::distance( first, last ) );
		}
	};

	template< typename T, typename H >
	struct hash< T, H, typename enable_if< is_string_base<T>::value, void >::type >
	{
		static H get( const T& value )
		{
			return value.template get_hash< H >();
		}
		inline H operator()( const T& value ) const { return get( value ); }
	};

	class string_view : public string_base< array_view< char > >
	{
	public:
		typedef string_base< array_view< char > > this_type;

		constexpr string_view() : this_type() {}
		inline string_view( const string_view& rhs ) : this_type( rhs.data(), rhs.size() ) {}
		template < typename S >
		inline string_view( const string_base<S>& rhs ) : this_type( rhs.data(), rhs.size() ) {}
		inline string_view( const std::string& str ) : this_type( str.data(), str.size() ) {}
		constexpr string_view( const char* str, size_type len ) : this_type( str, len ) {}

		// Literal constructors
		template< size_t N >
		constexpr string_view( char( &s )[N] ) : this_type( s, N - 1 ) {}
		template< size_t N >
		constexpr string_view( const char( &s )[N] ) : this_type( s, N - 1 ) {}

		// Non-literal constructors
		template< typename U >
		inline string_view( U str, typename enable_if<is_same<U, const char*>::value>::type* = nullptr ) : this_type( str, nvstrlen( str ) ) {}

		// Non-literal constructors
		template< typename U >
		inline string_view( U str, typename enable_if<is_same<U, char*>::value>::type* = nullptr ) : this_type( str, nvstrlen( str ) ) {}

		inline string_view& operator=( const string_view &rhs )
		{
			assign( rhs.data(), rhs.size() );
			return *this;
		}

		// modifiers
		inline void clear()
		{
			assign( nullptr, 0 );
		}
		inline void remove_prefix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data() + n, s - n );
		}
		inline void remove_suffix( size_type n )
		{
			size_type s = size();
			if ( n > s ) n = s;
			assign( data(), s - n );
		}

	};

	template < typename Storage >
	inline int string_base< Storage >::compare( const string_view& rhs ) const
	{
		size_type this_size = this->size();
		int cmp = nvmemcmp( this->data(), rhs.data(), nv::min( this_size, rhs.size() ) );
		return cmp != 0 ? cmp : ( this_size == rhs.size() ? 0 : this_size < rhs.size() ? -1 : 1 );
	}

	template < typename Storage >
	inline bool string_base< Storage >::starts_with( value_type c ) const
	{
		return !this->empty() && c == this->front();
	}
	template < typename Storage >
	inline bool string_base< Storage >::starts_with( const string_view& s ) const
	{
		return this->size() >= s.size() && nvmemcmp( this->data(), s.data(), s.size() ) == 0;
	}

	template < typename Storage >
	inline bool string_base< Storage >::ends_with( value_type c ) const
	{
		return !this->empty() && c == this->back();
	}
	template < typename Storage >
	inline bool string_base< Storage >::ends_with( const string_view& s ) const
	{
		return this->size() >= s.size() && nvmemcmp( this->data() + this->size() - s.size(), s.data(), s.size() ) == 0;
	}

	template < typename Storage >
	inline auto string_base< Storage >::find( value_type c, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_iterator it = nv::find_if( this->cbegin() + static_cast<difference_type>( pos ), this->cend(), [=] ( value_type val ) { return val == c; } );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find( const string_view& s, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_iterator it = nv::search( this->cbegin() + static_cast<difference_type>( pos ), this->cend(), s.cbegin(), s.cend() );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}

	template < typename Storage >
	inline auto string_base< Storage >::rfind( value_type c, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_reverse_iterator it = nv::find_if( this->crbegin() + static_cast<difference_type>( pos ), this->crend(), [=] ( value_type val ) { return val == c; } );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}
	template < typename Storage >
	inline auto string_base< Storage >::rfind( const string_view& s, size_type pos /*= 0*/ ) const -> size_type
	{
		if ( pos >= this->size() ) return npos;
		const_reverse_iterator it = nv::search( this->crbegin() + static_cast<difference_type>( pos ), this->crend(), s.crbegin(), s.crend() );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_first_of( value_type c ) const -> size_type
	{
		return this->find( c );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_first_of( const string_view& s ) const -> size_type
	{
		const_iterator it = nv::find_first_of( this->cbegin(), this->cend(), s.cbegin(), s.cend() );
		return it == this->cend() ? npos : static_cast<size_type>( nv::distance( this->cbegin(), it ) );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_last_of( value_type c ) const -> size_type
	{
		return this->rfind( c );
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_last_of( const string_view& s ) const -> size_type
	{
		const_reverse_iterator it = nv::find_first_of( this->crbegin(), this->crend(), s.cbegin(), s.cend() );
		return it == this->crend() ? npos : this->reverse_distance( this->crbegin(), it );
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_first_not_of( value_type c ) const -> size_type
	{
		for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
			if ( c != *it )
			return static_cast<size_type>( nv::distance( this->cbegin(), it ) );
		return npos;
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_first_not_of( const string_view& s ) const -> size_type
	{
		for ( const_iterator it = this->cbegin(); it != this->cend(); ++it )
			if ( 0 == nvmemchr( s.data(), static_cast<uchar8>( *it ), s.size() ) )
			return static_cast<size_type>( nv::distance( this->cbegin(), it ) );
		return npos;
	}

	template < typename Storage >
	inline auto string_base< Storage >::find_last_not_of( value_type c ) const -> size_type
	{
		for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
			if ( c != *it )
			return this->reverse_distance( this->crbegin(), it );
		return npos;
	}
	template < typename Storage >
	inline auto string_base< Storage >::find_last_not_of( const string_view& s ) const -> size_type
	{
		for ( const_reverse_iterator it = this->crbegin(); it != this->crend(); ++it )
			if ( 0 == nvmemchr( s.data(), static_cast<uchar8>( *it ), s.size() ) )
			return this->reverse_distance( this->crbegin(), it );
		return npos;
	}

	template < typename Storage >
	inline string_view string_base< Storage >::substr( size_type p, size_type n ) const
	{
		if ( p > this->size() ) return string_view(); // NV_THROW( out_of_range( "substr" ) );
		if ( n == p || p + n > this->size() ) n = this->size() - p;
		return string_view( this->data() + p, n );
	}

#define NV_STRING_BASE_CAST_OPERATORS( OPERATOR )\
template < typename S > inline bool operator OPERATOR ( const string_base< S >& lhs, const std::string& rhs )\
{\
	return lhs OPERATOR string_view( rhs );\
}\
template < typename S > inline bool operator OPERATOR ( const std::string& lhs, const string_base< S >& rhs )\
{\
	return string_view( lhs ) OPERATOR rhs;\
}\
template < typename S > inline bool operator OPERATOR ( const string_base< S >& lhs, const char* rhs )\
{\
	return lhs OPERATOR string_view( rhs );\
}\
template < typename S > inline bool operator OPERATOR ( const char* lhs, const string_base< S >& rhs )\
{\
	return string_view( lhs ) OPERATOR rhs;\
}\

	// Base operators
	template < typename S1, typename S2 >
	inline bool operator==( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.size() != rhs.size() ? false : ( lhs.compare( rhs ) == 0 );
	}
	template < typename S1, typename S2 >
	inline bool operator!=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.size() != rhs.size() ? true : ( lhs.compare( rhs ) != 0 );
	}
	template < typename S1, typename S2 >
	inline bool operator<( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) < 0;
	}
	template < typename S1, typename S2 >
	inline bool operator>( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) > 0;
	}
	template < typename S1, typename S2 >
	inline bool operator<=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) <= 0;
	}
	template < typename S1, typename S2 >
	inline bool operator>=( const string_base< S1 >& lhs, const string_base< S2 >& rhs )
	{
		return lhs.compare( rhs ) >= 0;
	}

	NV_STRING_BASE_CAST_OPERATORS( == )
		NV_STRING_BASE_CAST_OPERATORS( != )
		NV_STRING_BASE_CAST_OPERATORS( < )
		NV_STRING_BASE_CAST_OPERATORS( > )
		NV_STRING_BASE_CAST_OPERATORS( <= )
		NV_STRING_BASE_CAST_OPERATORS( >= )


#undef NV_STRING_REF_CAST_OPERATORS

	size_t sint32_to_buffer( sint32 n, char* str );
	size_t sint64_to_buffer( sint64 n, char* str );
	size_t uint32_to_buffer( uint32 n, char* str );
	size_t uint64_to_buffer( uint64 n, char* str );
	size_t f32_to_buffer( f32 n, char* str );
	size_t f64_to_buffer( f64 n, char* str );
	sint32 buffer_to_sint32( const char* str, char** end );
	sint64 buffer_to_sint64( const char* s, char** end );
	uint32 buffer_to_uint32( const char* s, char** end );
	uint64 buffer_to_uint64( const char* s, char** end );
	float buffer_to_f32( const char* s, char** end );
	double buffer_to_f64( const char* s, char** end );

	// const string is movable but not copyable
	class const_string : public string_base< array_view< char > >
	{
	public:
		inline const_string() {}
		inline const_string( const char* str, size_type len )
		{
			initialize( str, len );
		}
		inline explicit const_string( const char* str )
		{
			initialize( str, nvstrlen( str ) );
		}

		~const_string()
		{
			if ( data() )
			{
				delete data();
			}
		}

		inline const_string( const_string&& other )
		{
			assign( other.data(), other.size() );
			other.assign( nullptr, 0 );
		}

		inline const_string& operator=( const_string&& other )
		{
			pointer   old_data = data();
			size_type old_size = size();
			assign( other.data(), other.size() );
			other.assign( old_data, old_size );
			return *this;
		}

		// blocked copy constructor
		const_string( const const_string & ) = delete;
		// blocked copy operator
		const_string& operator=( const const_string& ) = delete;

	private:

		void initialize( const char* p, size_type s )
		{
			char* new_data = new char[s + 1];
			nvmemcpy( new_data, p, s );
			new_data[s] = 0;
			assign( new_data, s );
		}
	};

	class literal_string : public string_base< array_view< char > >
	{
		constexpr literal_string( const char* str, size_t len ) : this_type( str, len ) {}
	public:
		typedef string_base< array_view< char > > this_type;
		friend constexpr literal_string operator "" _ls( const char* str, size_t len );

		template< size_t N >
		constexpr literal_string( char( &s )[N] ) : this_type( s, N - 1 ) {}
		template< size_t N >
		constexpr literal_string( const char( &s )[N] ) : this_type( s, N - 1 ) {}
	};

	constexpr literal_string operator "" _ls( const char* str, size_t len )
	{
		return literal_string( str, len );
	}

	template < typename H >
	class hashed_literal_string;
	using hashed_literal_string_32 = hashed_literal_string< uint32 >;
	using hashed_literal_string_64 = hashed_literal_string< uint64 >;

	template < typename H >
	class hashed_literal_string : public string_base< array_view< char > >
	{
		constexpr hashed_literal_string( const char* str, size_t len, H hash_code ) 
			: this_type( str, len ), m_hash( hash_code ) {}
	public:
		typedef string_base< array_view< char > > this_type;

		template< size_t N >
		constexpr hashed_literal_string( char( &s )[N] )
			: this_type( s, N - 1 ), m_hash( detail::fnv_hash<H>::hash( s, len-1 ) ) {}
		template< size_t N >
		constexpr hashed_literal_string( const char( &s )[N] )
			: this_type( s, N - 1 ), m_hash( detail::fnv_hash<H>::hash( s, len - 1 ) ) {}

		template < typename H2 >
		constexpr H get_hash() const
		{
			static_assert( is_same< H, H2 >::value, "32/64 bit hash mismatch on hash_literal_string!" );
			return m_hash;
		}

		friend constexpr hashed_literal_string_32 operator "" _hls32( const char* str, size_t len );
		friend constexpr hashed_literal_string_64 operator "" _hls64( const char* str, size_t len );
	protected:
		H m_hash;
	};

	constexpr hashed_literal_string_32 operator "" _hls32( const char* str, size_t len )
	{
		return hashed_literal_string_32( str, len, detail::fnv_hash< uint32 >::hash( str, len ) );
	}

	constexpr hashed_literal_string_64 operator "" _hls64( const char* str, size_t len )
	{
		return hashed_literal_string_64( str, len, detail::fnv_hash< uint64 >::hash( str, len ) );
	}

	template< typename H >
	struct hash< hashed_literal_string_32, H >
	{
		static_assert( is_same< uint32, H >::value, "hashed_literal_string_32 used with non-32 bit hash!" );
		static constexpr H get( const hashed_literal_string_32& value )
		{
			return value.get_hash< H >();
		}
		constexpr H operator()( const hashed_literal_string_32& value ) const { return get( value ); }
	};

	template< typename H >
	struct hash< hashed_literal_string_64, H >
	{
		static_assert( is_same< uint64, H >::value, "hashed_literal_string_64 used with non-64 bit hash!" );
		static constexpr H get( const hashed_literal_string_64& value )
		{
			return value.get_hash< H >();
		}
		constexpr H operator()( const hashed_literal_string_64& value ) const { return get( value ); }
	};

	inline string_view trimmed( const string_view& str )
	{
		size_t endpos = str.find_last_not_of( " \r\n\t" );
		size_t startpos = str.find_first_not_of( " \r\n\t" );

		if ( string_view::npos != endpos || string_view::npos != startpos )
		{
			if ( string_view::npos == startpos ) startpos = 0;
			if ( string_view::npos != endpos )   endpos = endpos + 1 - startpos;
			return str.substr( startpos, endpos );
		}
		return str;
	}

	inline string_view rtrimmed( const string_view& str )
	{
		size_t endpos = str.find_last_not_of( " \r\n\t" );
		if ( string_view::npos != endpos )
		{
			return str.substr( 0, endpos + 1 );
		}
		return str;
	}

	inline string_view ltrimmed( const string_view& str )
	{
		size_t startpos = str.find_first_not_of( " \r\n\t" );
		if ( string_view::npos != startpos )
		{
			return str.substr( startpos );
		}
		return str;
	}


	inline string_view extract_extension( string_view filename )
	{
		size_t lastdot = filename.find_last_of( '.' );
		if ( string_view::npos != lastdot )
			return filename.substr( lastdot + 1 );
		return filename;
	}

}

using nv::operator "" _ls;
using nv::operator "" _hls32;
using nv::operator "" _hls64;

#endif // NV_STL_STRING_HH
