// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file raw.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief algorithm raw
*/

#ifndef NV_STL_ALGORITHM_RAW_HH
#define NV_STL_ALGORITHM_RAW_HH

#include <nv/stl/algorithm/common.hh>
#include <nv/base/capi.hh>

namespace nv
{
	
	// TODO: better debug macros, both here as well as in nv*funcs
	namespace detail
	{
		template< typename T >
		constexpr size_t byte_distance( const T* first, const T* last )
		{
			return static_cast<size_t>( reinterpret_cast<uintptr_t>( last ) - reinterpret_cast<uintptr_t>( first ) );
		}

	}

	template< typename T >
	T* raw_copy( const T* first, const T* last, T* out )
	{
		NV_DEBUG_ASSERT( last - first > 0, "raw_copy range fail!" );
		return static_cast<T*>( nvmemcpy( out, first, detail::byte_distance( first, last ) ) ) + ( last - first );
	}

	template< typename T >
	T* raw_copy_n( const T* ptr, size_t n, T* out )
	{
		return static_cast<T*>( nvmemcpy( out, ptr, n * sizeof( T ) ) ) + n;
	}

	template< typename T >
	T* raw_alias_copy( const T* first, const T* last, T* out )
	{
		NV_DEBUG_ASSERT( last - first > 0, "raw_alias_copy range fail!" );
		return static_cast<T*>( nvmemmove( out, first, detail::byte_distance( first, last ) ) ) + ( last - first );
	}

	template< typename T >
	T* raw_alias_copy_n( const T* ptr, size_t n, T* out )
	{
		return static_cast<T*>( nvmemmove( out, ptr, n * sizeof( T ) ) ) + n;
	}

	template< typename T >
	T* raw_zero( T* first, T* last )
	{
		NV_DEBUG_ASSERT( last - first > 0, "raw_zero range fail!" );
		return static_cast<T*>( nvmemset( first, 0, detail::byte_distance( first, last ) ) ) + ( last - first );
	}

	template< typename T >
	T* raw_zero_n( T* ptr, size_t n )
	{
		return static_cast<T*>( nvmemset( ptr, 0, n * sizeof( T ) ) ) + n;
	}

	template< typename T >
	T* raw_fill( T* first, T* last, unsigned char value )
	{
		NV_DEBUG_ASSERT( last - first > 0, "raw_fill range fail!" );
		return static_cast<T*>( nvmemset( first, value, detail::byte_distance( first, last ) ) ) + ( last - first );
	}

	template< typename T >
	T* raw_fill_n( T* ptr, size_t n, unsigned char value )
	{
		return static_cast<T*>( nvmemset( ptr, value, n * sizeof( T ) ) ) + n;
	}

}

#endif // NV_STL_ALGORITHM_RAW_HH

