// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file library.hh
 * @author Kornel Kisielewicz
 * @brief Implements a dynamic library class
 */

#ifndef NV_CORE_LIBRARY_HH
#define NV_CORE_LIBRARY_HH

#include <nv/common.hh>
#include <nv/stl/exception.hh>
#include <nv/stl/string.hh>
#include <string>

namespace nv
{
	/**
	 * library
	 * @brief Class representing a dynamically loaded library
	 */
	class library 
	{
	public:

		/**
		 * Library constructor
		 */
		library();

		/**
		 * Opens the library
		 *
		 * Throws library_error on failure
		 */
		void open( string_view name );

		/**
		 * Tries to open the library
		 *
		 * returns true if succeeded, false otherwise
		 */
		bool try_open( string_view name );

		/**
		 * Returns true if the library is open, false otherwise
		 */
		bool is_open() const;

		/**
		 * Returns library name
		 */
		string_view get_name() const;

		/**
		 * Get symbol from library
		 *
		 * Throws on symbol not found
		 */
		void* get( string_view symbol );

		/**
		 * Get symbol from library
		 *
		 * Returns null if symbol not found
		 */
		void* try_get( string_view symbol );

		/**
		 * Destructor
		 *
		 * Closes the library if open.
		 */
		~library();

		/**
		 * Returns library loading error if present
		 *
		 * Exact implementation depends on platform/compiler.
		 */
		static std::string get_error();

	protected:
		/**
		 * Opens the library
		 *
		 * Opens the library and prepares it for getting symbols.
		 * Needs to be done before any get call.
		 *
		 * returns true on success, false otherwise
		 */
		bool open();

		/**
		 * @brief Closes the library
		 *
		 * Closes the library. Any further get calls or operations on
		 * received symbols are invalid!
		 */
		void close();

		/// Library handle
		void* m_handle;

		/// Library name
		std::string m_name;

	};  // class Library

	class library_error : public runtime_error
	{
		/// Library name
		std::string m_name;
	public:
		/**
		 * Constructor
		 */
		library_error( const std::string& message, const std::string& name )
			: runtime_error( "Library (" + name + ") : " + message + " [ " + library::get_error() + " ]"), m_name( name )
		{
		}

		/**
		 * Returns library name
		 */
		const std::string& get_name()
		{
			return m_name;
		}
	};

} // namespace nv

#endif // NV_CORE_LIBRARY_HH
