// Copyright (C) 2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
* @file contiguous_storage_policy.hh
* @author Kornel Kisielewicz epyon@chaosforge.org
* @brief contiguous storage policy classes
*/

#ifndef NV_STL_CONTAINER_CONTIGUOUS_STORAGE_POLICY
#define NV_STL_CONTAINER_CONTIGUOUS_STORAGE_POLICY

#include <nv/common.hh>
#include <nv/stl/memory.hh>
#include <nv/stl/container/contiguous_storage.hh>

namespace nv
{

	template< typename Storage, size_t N >
	class fixed_storage : public Storage
	{
	public:
		typedef size_t                       size_type;
		typedef typename Storage::value_type value_type;

		static constexpr bool is_fixed = true;

		fixed_storage()
		{
			Storage::reallocate( N, false );
		}
		~fixed_storage()
		{
			Storage::reallocate( 0, false );
		}
		static constexpr size_type max_size() { return N; }
		static constexpr size_type capacity() { return N; }
		static constexpr size_type size() { return N; }
		static constexpr bool empty() { return N == 0; }
		static constexpr size_type raw_size() { return sizeof( value_type ) * N; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), size() ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), size() ); }

		// allow move
		fixed_storage( fixed_storage&& ) = default;
		fixed_storage& operator=( fixed_storage&& ) = default;
	};

	template< typename Storage >
	class resizable_storage : public Storage
	{
	public:
		typedef size_t                       size_type;
		typedef typename Storage::value_type value_type;

		static constexpr bool is_fixed = false;

		~resizable_storage()
		{
			if ( m_size > 0 ) Storage::reallocate( 0, false );
		}
		static constexpr size_type max_size() { return size_type( 0x80000000 ); }
		constexpr size_t capacity() { return m_size; }
		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr size_t raw_size() const { return sizeof( value_type ) * m_size; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), size() ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), size() ); }
	protected:
		constexpr resizable_storage() : m_size( 0 ) {}

		// allow move
		inline resizable_storage( resizable_storage&& other )
			: Storage( nv::move( other ) ), m_size( other.m_size )
		{
			other.m_size = 0;
		}
		inline resizable_storage& operator=( resizable_storage&& other )
		{
			m_size = other.m_size;
			Storage::operator=( nv::move( other ) );
			other.m_size = 0;
			return *this;
		}

		// TODO: return type error checking
		bool try_resize( size_t new_size, bool copy_needed )
		{
			if ( new_size != m_size )
			{
				if ( Storage::reallocate( new_size, copy_needed ) )
				{
					m_size = new_size;
					return true;
				}
				return false;
			}
			return true;
		}
	protected:
		size_type m_size;
	};

	template< typename SizeType >
	struct default_next_capacity
	{
		static SizeType get( SizeType requested, SizeType capacity, SizeType max_size )
		{
			SizeType minimum = nv::min<SizeType>( capacity, 4 );
			SizeType remaining = max_size - capacity;
			if ( remaining < requested ) return 0;
			SizeType additional = nv::max( requested, capacity );
			return nv::max( minimum, remaining < additional ? max_size : capacity + additional );
		}
	};

	template< typename Storage, typename NextCapacity = default_next_capacity< size_t > >
	class growable_storage : public Storage
	{
	public:
		typedef size_t                       size_type;
		typedef typename Storage::value_type value_type;

		static constexpr bool is_fixed = false;

		~growable_storage()
		{
			if ( m_capacity > 0 ) Storage::reallocate( 0, false );
		}
		static constexpr size_type max_size() { return size_type( 0x80000000 ); }
		constexpr size_t capacity() { return m_capacity; }
		constexpr size_t size() const { return m_size; }
		constexpr bool empty() const { return m_size == 0; }
		constexpr size_t raw_size() const { return sizeof( value_type ) * m_size; }

		operator array_ref< value_type >() { return array_ref< value_type >( Storage::data(), size() ); }
		operator array_view< value_type >() const { return array_view< value_type >( Storage::data(), size() ); }
	protected:
		constexpr growable_storage() : m_size( 0 ), m_capacity( 0 ) {}

		// allow move
		inline growable_storage( growable_storage&& other )
			: Storage( nv::move( other ) ), m_size( other.m_size ), m_capacity( other.m_capacity )
		{
			other.m_size = 0;
			other.m_capacity = 0;
		}
		inline growable_storage& operator=( growable_storage&& other )
		{
			m_size = other.m_size;
			m_capacity = other.m_capacity;
			Storage::operator=( nv::move( other ) );
			other.m_size = 0;
			other.m_capacity = 0;
			return *this;
		}

		// TODO: return type error checking
		bool try_grow( size_t amount )
		{
			size_type new_size = amount + m_size;
			if ( new_size > m_capacity )
			{
				size_type new_capacity = NextCapacity::get( new_size - m_capacity, m_capacity, max_size() );
				if ( new_capacity > 0 && Storage::reallocate( new_capacity, true ) )
				{
					m_capacity = new_capacity;
					m_size = new_size;
				}
				else return false;
			}
			m_size = new_size;
			return true;
		}
		// TODO: return type error checking
		bool try_reserve( size_t new_capacity, bool copy_needed )
		{
			if ( new_capacity > m_capacity )
			{
				if ( Storage::reallocate( new_capacity, copy_needed ) )
				{
					m_capacity = new_capacity;
				}
				else return false;
			}
			return true;
		}
		// TODO: return type error checking
		bool try_resize( size_t new_size, bool copy_needed )
		{
			if ( new_size > m_size )
			{
				if ( try_reserve( new_size, copy_needed ) )
				{
					m_size = new_size;
					return true;
				}
				return false;
			}
			m_size = new_size;
			return true;
		}
	protected:
		size_type m_size;
		size_type m_capacity;
	};

}

#endif // NV_STL_CONTAINER_CONTIGUOUS_STORAGE_POLICY