// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

#include "nv/core/library.hh"

#if NV_PLATFORM == NV_WINDOWS
#   define WIN32_LEAN_AND_MEAN
#   include <windows.h>
#   define NV_LIB_EXT ".dll"
#   define NV_LIB_HANDLE HMODULE
#   define NV_LIB_OPEN( name ) LoadLibraryEx( name, NULL, LOAD_WITH_ALTERED_SEARCH_PATH )
#   define NV_LIB_GET( handle, name ) GetProcAddress( handle, name )
#   define NV_LIB_CLOSE( name ) !FreeLibrary( name )
#elif NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
#   include <dlfcn.h>
#   define NV_LIB_EXT ".so"
#   define NV_LIB_HANDLE void*
#   define NV_LIB_OPEN( name ) dlopen( name, RTLD_LAZY | RTLD_GLOBAL)
#   define NV_LIB_GET( handle, name ) dlsym( handle, name )
#   define NV_LIB_CLOSE( name ) dlclose( name )
#elif NV_PLATFORM == NV_APPLE
#   include "macUtils.h"
#   include <dlfcn.h>
#   define NV_LIB_EXT ".dylib"
#   define NV_LIB_HANDLE CFBundleRef
#   define NV_LIB_OPEN( name ) mac_loadExeBundle( name )
#   define NV_LIB_GET( handle, name ) mac_getBundleSym( handle, name )
#   define NV_LIB_CLOSE( name ) mac_unloadExeBundle( name )
#endif

#include "nv/core/logging.hh"

using namespace nv;

library::library() 
    : m_handle( nullptr ), m_name()
{
}

void library::open( string_view name )
{
	m_name.assign( name.data(), name.size() );
	if ( !open() )
	{
		m_handle = nullptr;
		NV_THROW( library_error, "Can't load library!", name.data() );
	}
}

bool nv::library::try_open( string_view name )
{
	m_name.assign( name.data(), name.size() );
	if ( !open() )
	{
		m_handle = nullptr;
		return false;
	}
	return true;
}

string_view library::get_name() const
{
    return string_view( m_name );
}

bool library::open( )
{
    if ( m_handle != NULL )
    {
        return true;
    }
    NV_LOG_NOTICE( "library : loading '", m_name, "'..." );

	std::string name = m_name;
	string_view ext( NV_LIB_EXT );

	if ( name.length() < ext.length() || name.substr( name.length() - ext.length(), ext.length() ) != ext )
    {
        name.append( ext.data(), ext.length() );
    }

    m_handle = (void*)NV_LIB_OPEN( name.c_str() );

    if ( m_handle == NULL )
    {
		NV_LOG_NOTICE( "library : '", name, "' failed to open." );
		return false;
    }
    NV_LOG_NOTICE( "library : '", name, "' loaded." );
	return true;
}

void* library::get( string_view symbol )
{
	void* result = (void*) NV_LIB_GET( (NV_LIB_HANDLE) m_handle, symbol.data() );
    if ( !result )
    {
        NV_THROW( library_error, "Can't find symbol " + std::string(symbol.data(),symbol.size()) + "!", m_name );
    }
	return result;
}

void* nv::library::try_get( string_view symbol )
{
	return (void*) NV_LIB_GET( (NV_LIB_HANDLE) m_handle, symbol.data() );
}

bool library::is_open() const
{
	return m_handle != nullptr;
}

void library::close()
{
    if ( NV_LIB_CLOSE( (NV_LIB_HANDLE)m_handle ) )
    {
        NV_LOG_ERROR( "library : can't close library '", m_name, "'!" );
    }
    m_handle = NULL;
}

library::~library()
{
    if ( m_handle != NULL )
    {
        close();
    }
}

std::string library::get_error()
{
#if NV_PLATFORM == NV_WINDOWS
    // We do hate WinAPI for code like this, don't we?
    LPTSTR buffer = NULL;
    FormatMessage( FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM | FORMAT_MESSAGE_IGNORE_INSERTS,
        NULL, GetLastError(), MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), (LPTSTR) &buffer, 0, NULL );
    std::string msg( (char*)buffer );
    LocalFree( buffer );
    return msg;
#elif NV_PLATFORM == NV_LINUX || NV_PLATFORM == NV_APPLE
    return string(dlerror());
#else
    return string("");
#endif
}
