// Copyright (C) 2012-2015 ChaosForge Ltd
// http://chaosforge.org/
//
// This file is part of Nova libraries. 
// For conditions of distribution and use, see copying.txt file in root folder.

/**
 * @file logging.hh
 * @author Kornel Kisielewicz epyon@chaosforge.org
 * @brief logging interfaces
 */

#ifndef NV_CORE_LOGGING_HH
#define NV_CORE_LOGGING_HH

#include <nv/common.hh>
#include <nv/stl/capi.hh>
#include <nv/stl/string.hh>
#include <nv/stl/singleton.hh>

namespace nv
{

	enum log_level
	{
		LOG_NONE     = 0,
		LOG_FATAL    = 10,
		LOG_CRITICAL = 20,
		LOG_ERROR    = 30,
		LOG_WARNING  = 40,
		LOG_NOTICE   = 50,
		LOG_INFO     = 60,
		LOG_DEBUG    = 80,
		LOG_TRACE    = 100
	};

	class logger_base : public singleton< logger_base >
	{
	public:
		logger_base()
		{
			m_pos = m_message;
		}
		unsigned int get_level()
		{
			return m_level;
		}
		void set_level( unsigned int level )
		{
			m_level = level;
		}
		virtual void log( log_level level, const string_ref& message ) = 0;
		void log_append( string_ref ref )
		{
			nvmemcpy( m_pos, ref.data(), ref.size() );
			m_pos += ref.size();
		}
		void log_append( uint32 u )
		{
			m_pos += uint32_to_buffer( u, m_pos );
		}
		void log_append( sint32 s )
		{
			m_pos += sint32_to_buffer( s, m_pos );
		}
		void log_append( uint64 u )
		{
			m_pos += uint64_to_buffer( u, m_pos );
		}
		void log_append( sint64 s )
		{
			m_pos += sint64_to_buffer( s, m_pos );
		}
		void log_append( f32 f )
		{
			m_pos += f32_to_buffer( f, m_pos );
		}
		static bool can_log( log_level level )
		{
			return logger_base::is_valid() && (unsigned int)reference().get_level() >= (unsigned int)level;
		}

		void log_sequence( log_level level )
		{
			*m_pos = '\0';
			log( level, string_ref( m_message, (size_t)( m_pos - m_message ) ) );
			m_pos = m_message;
		}
		template < typename T, typename... Args >
		inline void log_sequence( log_level level, T&& t, Args&&... args )
		{
			log_append( t );
			log_sequence( level, nv::forward<Args>( args )... );
		}
		virtual ~logger_base() {}
	protected:
		char m_message[256];
		char* m_pos;
		unsigned int m_level;
	};

} // namespace nv

#define NV_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
	{\
		nv::logger_base::reference().log_sequence( level, __VA_ARGS__ ); \
	}

#if NV_DEBUG == 1
#define NV_DEBUG_LOG(level, ...) \
	if ( nv::logger_base::can_log(level) ) \
	{\
		nv::logger_base::reference().log_sequence( level, __VA_ARGS__ ); \
	}
#else
#define NV_DEBUG_LOG(level, ...)
#endif

#define NV_LOG_FATAL(...)    NV_LOG( nv::LOG_FATAL, __VA_ARGS__ )
#define NV_LOG_CRITICAL(...) NV_LOG( nv::LOG_CRITICAL, __VA_ARGS__ )
#define NV_LOG_ERROR(...)    NV_LOG( nv::LOG_ERROR, __VA_ARGS__ )
#define NV_LOG_WARNING(...)  NV_LOG( nv::LOG_WARNING, __VA_ARGS__ )
#define NV_LOG_NOTICE(...)   NV_LOG( nv::LOG_NOTICE, __VA_ARGS__ )
#define NV_LOG_INFO(...)     NV_LOG( nv::LOG_INFO, __VA_ARGS__ )
#define NV_LOG_DEBUG(...)    NV_LOG( nv::LOG_DEBUG, __VA_ARGS__ )
#define NV_LOG_TRACE(...)    NV_LOG( nv::LOG_TRACE, __VA_ARGS__ )

#endif // NV_CORE_LOGGING_HH
